/*
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 */
#include "upm_commondbus.h"

Q_GLOBAL_STATIC(UpmCommonDBus, s_commonDBus)

UpmCommonDBus::UpmCommonDBus(QObject *parent) : QObject(parent)
{
    //systemd login1 dbus
    QDBusConnection::systemBus().connect(
                "org.freedesktop.login1",
                "/org/freedesktop/login1",
                "org.freedesktop.login1.Manager",
                "PrepareForSleep",
                this,
                SLOT(dealPrepareForSleep(bool)));
    //system idle status
    QDBusConnection::sessionBus().connect(
                "org.gnome.SessionManager",
                "/org/gnome/SessionManager/Presence",
                "org.gnome.SessionManager.Presence",
                "StatusChanged",
                this,
                SLOT(dealSystemStatusChanged(quint32)));
    //user active signal
    QDBusConnection::sessionBus().connect("org.ukui.SettingsDaemon",
                                          "/GlobaSignal",
                                          "org.ukui.SettingsDaemon.GlobalSignal",
                                          "Active",
                                          this,
                                          SLOT(dealUserActiveSignal(bool)));
    initPlatformType();
}

UpmCommonDBus::~UpmCommonDBus() {}

UpmCommonDBus *UpmCommonDBus::self()
{
    return s_commonDBus;
}

void UpmCommonDBus::dealPrepareForSleep(bool isSleep)
{
    emit systemSleepStatus(isSleep);
}

void UpmCommonDBus::dealSystemStatusChanged(quint32 status)
{
    qDebug() << "Idle signal received from Session Manager" << status;
    if (SYSTEM_IDEL_SIGNAL_VALUE == status) {
        emit systemIdleStatus(true);
    } else {
        emit systemIdleStatus(false);
    }
}

void UpmCommonDBus::dealUserActiveSignal(bool active)
{
    emit userActiveSignal(active);
}

quint32 UpmCommonDBus::sendNotification(
        const QString &type,
        const QString &arg,
        bool permanent)
{
    QDBusInterface dbusInterface("org.freedesktop.Notifications",
                                 "/org/freedesktop/Notifications",
                                 "org.freedesktop.Notifications",
                                 QDBusConnection::sessionBus());
    QList<QVariant> args;
    QStringList argg;
    QMap<QString, QVariant> pear_map;

    if (true == permanent) {
        pear_map.insert("x-ukui-popup-timeout", -1);
    }

    args << tr("Power Manager") << ((unsigned int)0) << QString("ukui-power-manager")
         << type //显示的是什么类型的信息
         << arg  //显示的具体信息
         << argg << pear_map << 0;

    QDBusReply<quint32> reply = dbusInterface.callWithArgumentList(QDBus::AutoDetect, "Notify", args);
    qDebug() << "notification id is:" << reply.value();
    return reply.value();
}

void UpmCommonDBus::closeNotification(quint32 notificationId)
{
    qDebug() << "close notification id is:" << notificationId;
    QDBusInterface dbusInterface("org.freedesktop.Notifications",
                                 "/org/freedesktop/Notifications",
                                 "org.freedesktop.Notifications",
                                 QDBusConnection::sessionBus());
    dbusInterface.call("CloseNotification", notificationId);
}

void UpmCommonDBus::initPlatformType()
{
    QString cloudPlatForm = QString(QLatin1String(""));
    QString hostType = QString(QLatin1String(""));
#ifdef KY_SDK_SYSINFO
    cloudPlatForm = QString(QLatin1String(kdk_system_get_hostCloudPlatform()));
    hostType = QString(QLatin1String(kdk_system_get_hostVirtType()));
#endif

    qDebug() << "cloud platform name:" << cloudPlatForm
             << "host type:" << hostType;
    if ((cloudPlatForm == "huawei" || cloudPlatForm == "ctyun" || hostType != "none")) {
        m_isCloudPlatform = true;
        qDebug() << "is cloud platform";
    }
}

bool UpmCommonDBus::isCloudPlatform()
{
    return m_isCloudPlatform;
}

/*
void UpmCommonDBus::playSystemSound(SoundType type)
{
    KSoundEffects::playSound(type);
}
*/
