// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package path

import (
	"path/filepath"
	"strings"

	"code.forgejo.org/f3/gof3/v3/id"
	"code.forgejo.org/f3/gof3/v3/kind"
)

func PathAbsoluteString(current, destination string) string {
	if !strings.HasPrefix(destination, "/") {
		return filepath.Clean(current + "/" + destination)
	}
	return filepath.Clean(destination)
}

func PathAbsolute(allocator PathElementAllocator, current, destination string) Path {
	return NewPathFromString(allocator, PathAbsoluteString(current, destination))
}

func PathRelativeString(current, destination string) string {
	r, err := filepath.Rel(current, destination)
	if err != nil {
		panic(err)
	}
	return r
}

func NewPathFromString(allocator PathElementAllocator, pathString string) Path {
	pathString = filepath.Clean(pathString)
	if pathString == "." {
		e := allocator()
		e.SetID(id.NewNodeID("."))
		return NewPath(e)
	}
	path := make([]PathElement, 0, 10)
	if strings.HasPrefix(pathString, "/") {
		root := allocator()
		root.SetKind(kind.KindRoot)
		path = append(path, root)
		pathString = pathString[1:]
	}
	if pathString == "" {
		return NewPath(path...)
	}
	for _, i := range strings.Split(pathString, "/") {
		e := allocator()
		e.SetID(id.NewNodeID(i))
		path = append(path, e)
	}
	return NewPath(path...)
}

func NewPath(nodes ...PathElement) Path {
	return Implementation(nodes)
}
