// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// Copyright twenty-panda <twenty-panda@posteo.com>
// SPDX-License-Identifier: MIT

package gitlab

import (
	"context"
	"fmt"
	"net/http"

	"code.forgejo.org/f3/gof3/v3/f3"
	"code.forgejo.org/f3/gof3/v3/id"
	f3_tree "code.forgejo.org/f3/gof3/v3/tree/f3"
	"code.forgejo.org/f3/gof3/v3/tree/generic"
	"code.forgejo.org/f3/gof3/v3/util"

	"gitlab.com/gitlab-org/api/client-go"
)

type organization struct {
	common
	forgejoOrganization *gitlab.Group
}

var _ f3_tree.ForgeDriverInterface = &organization{}

func newOrganization() generic.NodeDriverInterface {
	return &organization{}
}

func (o *organization) SetNative(organization any) {
	o.forgejoOrganization = organization.(*gitlab.Group)
}

func (o *organization) GetNativeID() string {
	return fmt.Sprintf("%d", o.forgejoOrganization.ID)
}

func (o *organization) NewFormat() f3.Interface {
	node := o.GetNode()
	return node.GetTree().(f3_tree.TreeInterface).NewFormat(node.GetKind())
}

func (o *organization) ToFormat() f3.Interface {
	if o.forgejoOrganization == nil {
		return o.NewFormat()
	}
	return &f3.Organization{
		Common:   f3.NewCommon(fmt.Sprintf("%d", o.forgejoOrganization.ID)),
		Name:     o.forgejoOrganization.Name,
		FullName: o.forgejoOrganization.Description,
	}
}

func (o *organization) FromFormat(content f3.Interface) {
	organization := content.(*f3.Organization)
	o.forgejoOrganization = &gitlab.Group{
		ID:          int(util.ParseInt(organization.GetID())),
		Name:        organization.Name,
		Path:        organization.Name,
		Description: organization.FullName,
	}
}

func (o *organization) Get(ctx context.Context) bool {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	organization, resp, err := o.getClient().Groups.GetGroup(node.GetID().Int(), &gitlab.GetGroupOptions{})
	if resp.StatusCode == http.StatusNotFound {
		return false
	}
	if err != nil {
		panic(fmt.Errorf("organization %v %w", o, err))
	}
	o.forgejoOrganization = organization
	return true
}

func (o *organization) Patch(context.Context) {
	node := o.GetNode()
	o.Trace("%s", node.GetID())
	_, _, err := o.getClient().Groups.UpdateGroup(o.forgejoOrganization.Name, &gitlab.UpdateGroupOptions{
		Description: &o.forgejoOrganization.Description,
	})
	if err != nil {
		panic(fmt.Errorf("UpdateGroup %v %w", o, err))
	}
}

func (o *organization) Put(context.Context) id.NodeID {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	organization, _, err := o.getClient().Groups.CreateGroup(&gitlab.CreateGroupOptions{
		Name:        &o.forgejoOrganization.Name,
		Path:        &o.forgejoOrganization.Name,
		Description: &o.forgejoOrganization.Description,
	})
	if err != nil {
		panic(fmt.Errorf("CreateGroup %v %w", o, err))
	}
	o.forgejoOrganization = organization
	o.Trace("%s", organization)
	return id.NewNodeID(o.GetNativeID())
}

func (o *organization) Delete(ctx context.Context) {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	permanentlyRemove := true
	_, err := o.getClient().Groups.DeleteGroup(o.forgejoOrganization.Name, &gitlab.DeleteGroupOptions{
		PermanentlyRemove: &permanentlyRemove,
	})
	if err != nil {
		panic(fmt.Errorf("DeleteGroup %v %w", o, err))
	}
}
