/*
	Description: macro handling

	Author: Marco Costalba (C) 2006

	Copyright: See COPYING file that comes with this distribution

*/
#include <qsettings.h>
#include <qlistview.h>
#include <qinputdialog.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qtextedit.h>
#include <qcheckbox.h>
#include "common.h"
#include "macroimpl.h"

using namespace QGit;

MacroImpl::MacroImpl() : Macro(0, 0, Qt::WDestructiveClose) {

	listViewNames->setSorting(-1);

	QSettings set;
	macroList = QStringList::split(",", set.readEntry(APP_KEY + MCR_LIST_KEY, ""));
	QListViewItem* lastItem = NULL;
	loopList(it, macroList)
		lastItem = new QListViewItem(listViewNames, lastItem, *it);

	listViewNames_currentChanged(listViewNames->currentItem());
	if (listViewNames->currentItem())
		listViewNames->currentItem()->setSelected(true);
}

void MacroImpl::loadMacro(const QString& macroName) {

	checkBoxRefreshAfterMacro->setChecked(testFlag(MCR_REFRESH_F, macroName));
	checkBoxAskArgs->setChecked(testFlag(MCR_CMD_LINE_F, macroName));
	QSettings set;
	textEditMacro->setText(set.readEntry(APP_KEY + macroName + MCR_TEXT_KEY, ""));
}

void MacroImpl::removeMacro(const QString& macroName) {

	QSettings set;
	set.removeEntry(APP_KEY + macroName + FLAGS_KEY);
	set.removeEntry(APP_KEY + macroName + MCR_TEXT_KEY);
}

void MacroImpl::updateMacroList() {

	macroList.clear();
	QListViewItemIterator it(listViewNames);
	while (it.current()) {
		macroList.append(it.current()->text(0));
		++it;
	}
	writeSetting(MCR_LIST_KEY, macroList.join(","));
	emit macroListChanged(macroList);
}

void MacroImpl::listViewNames_currentChanged(QListViewItem* item) {

	bool emptyList = (item == NULL);

	if (!emptyList) {
		curMacro = "Macro " + item->text(0) + "/";
		loadMacro(curMacro);
		listViewNames->ensureItemVisible(item);
	} else {
		curMacro = "";
		textEditMacro->clear();
		if (checkBoxRefreshAfterMacro->isChecked())
			checkBoxRefreshAfterMacro->toggle();
		if (checkBoxAskArgs->isChecked())
			checkBoxAskArgs->toggle();
	}

	textEditMacro->setEnabled(!emptyList);
	checkBoxRefreshAfterMacro->setEnabled(!emptyList);
	checkBoxAskArgs->setEnabled(!emptyList);
	pushButtonRename->setEnabled(!emptyList);
	pushButtonRemove->setEnabled(!emptyList);
	pushButtonMoveUp->setEnabled(!emptyList && (item != listViewNames->firstChild()));
	pushButtonMoveDown->setEnabled(!emptyList && (item != listViewNames->lastItem()));
}

bool MacroImpl::getNewName(QString& name, const QString& caption) {

	bool ok;
	const QString oldName = name;
	name = QInputDialog::getText(caption + " - QGit", "Enter action name:",
	                             QLineEdit::Normal, name, &ok, this);
	if (!ok || name.isEmpty() || name == oldName)
		return false;

	if (macroList.contains(name)) {
		QMessageBox::warning(this, caption + " - QGit", "Sorry, action name "
		                     "already exists.\nPlease choose a different name.");
		return false;
	}
	return true;
}

void MacroImpl::pushButtonNew_clicked() {

	QString name;
	if (!getNewName(name, "Create new action"))
		return;

	QListViewItem* item = new QListViewItem(listViewNames, listViewNames->lastItem(), name);
	listViewNames->setCurrentItem(item);
	listViewNames_currentChanged(listViewNames->currentItem());
	updateMacroList();
}

void MacroImpl::pushButtonRename_clicked() {

	QListViewItem* item = listViewNames->currentItem();
	if (!item || !item->isSelected())
		return;

	QString newName(item->text(0));
	if (!getNewName(newName, "Rename action"))
		return;

	item->setText(0, newName);
	updateMacroList();
	const QString oldMacroName(curMacro);
	listViewNames_currentChanged(listViewNames->currentItem()); // updates curMacro
	loadMacro(oldMacroName);
	removeMacro(oldMacroName);
}

void MacroImpl::pushButtonRemove_clicked() {

	QListViewItem* item = listViewNames->currentItem();
	if (!item || !item->isSelected())
		return;

	removeMacro(curMacro);
	delete item;
	updateMacroList();
	if (listViewNames->currentItem())
		listViewNames->currentItem()->setSelected(true);
}

void MacroImpl::pushButtonMoveUp_clicked() {

	QListViewItem* item = listViewNames->currentItem();
	if (!item || item == listViewNames->firstChild())
		return;

	item->itemAbove()->moveItem(item);
	updateMacroList();
	listViewNames_currentChanged(item);
}

void MacroImpl::pushButtonMoveDown_clicked() {

	QListViewItem* item = listViewNames->currentItem();
	if (!item || item == listViewNames->lastItem())
		return;

	item->moveItem(item->itemBelow());
	updateMacroList();
	listViewNames_currentChanged(item);
}

void MacroImpl::textEditMacro_textChanged() {

	if (!curMacro.isEmpty())
		writeSetting(MCR_TEXT_KEY, textEditMacro->text(), curMacro);
}

void MacroImpl::checkBoxRefreshAfterMacro_toggled(bool b) {

	if (!curMacro.isEmpty())
		setFlag(MCR_REFRESH_F, b, curMacro);
}

void MacroImpl::checkBoxAskArgs_toggled(bool b) {

	if (!curMacro.isEmpty())
		setFlag(MCR_CMD_LINE_F, b, curMacro);
}

void MacroImpl::pushButtonOk_clicked() {

	close();
}
