#!/usr/bin/env python
# arch-tag: c821e9df-8e3d-4728-838a-0d78775eb4c6
# Copyright (C) 2004 David Allouche <david@allouche.net>
#                    Canonical Ltd.
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test cases for name escaping.
"""

import sys
import unittest
import pybaz as arch
from pybaz.tests import framework


class NameEscaping(unittest.TestCase):

    tests = []

    def escaped_charset(self, null_too):
        def uni_escape(start, end):
            return ''.join(['\\(U+%X)' % C for C in range(start, end+1)])
        charset = ''.join((
            uni_escape(0x1, 0x8),
            '\\(tab)\\(nl)\\(U+B)\\(np)\\(cr)',
            uni_escape(0xE, 0x1F),
            '\\(sp)!"#$%&\'()*+,-./0123456789:;<=>?@',
            'ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\\\]^_`',
            'abcdefghijklmnopqrstuvwxyz{|}~',
            '\x7f', ### That looks like  a bug in tla
            uni_escape(0x80, 0xFF)))
        if null_too: charset = uni_escape(0,0) + charset
        return charset

    def help_test_escape(self, command, null_too):
        if null_too: start = 0
        else: start = 1
        charset = ''.join([chr(C) for C in range(start, 256)])
        expected = self.escaped_charset(null_too)
        result = command(charset)
        self.assertEqual(expected, result, False)

    def tla_escape(self):
        """tla escape does what is expected."""
        from pybaz.backends import baz
        ### XXX disable test case if tla does not support escaping
        if 1 == baz.status_cmd(['escape'], (0,1)): return
        ### XXXX
        self.help_test_escape(baz.tla_name_escape, False)
    tests.append('tla_escape')

    def python_escape(self):
        """Native escaping works correctly."""
        self.help_test_escape(arch.name_escape, True)
    tests.append('python_escape')

    def interesting_escapes(self):
        def symbols(syms):
            return ''.join(['\\(%s)' % S for S in syms])
        return symbols(('TaB', 'NL', 'Np', 'Cr', 'SP')) + \
               '\\\\' '\\"' '\\(U+4A)\\(u+4B)\\(U+4c)\\(U+00004D)'

    def help_test_unescape(self, command, null_too):
        if null_too: start = 0
        else: start = 1
        expected = ''.join([chr(C) for C in range(start, 256)])
        result = command(self.escaped_charset(null_too))
        self.assertEqual(expected, result) # whole charset
        expected = ''.join(map(chr, [9, 10, 12, 13, 32]) + ['\\', '"', 'JKLM'])
        result = command(self.interesting_escapes())
        self.assertEqual(expected, result) # interesting escapes

    def tla_unescape(self):
        """tla unescape does what is expected."""
        from pybaz.backends import baz
        ### XXX disable test case if tla does not support escaping
        if 1 == baz.status_cmd(['escape'], (0,1)): return
        ### XXX
        self.help_test_unescape(baz.tla_name_unescape, False)
    tests.append('tla_unescape')

    def python_unescape(self):
        """Native unescaping works correctly."""
        self.help_test_unescape(arch.name_unescape, False)
    tests.append('python_unescape')


framework.register(__name__)
