/*
 *  Copyright 2001-2005 Internet2
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* SAMLResponse.cpp - SAML response implementation

   Scott Cantor
   5/28/02

   $History:$
*/

#include "internal.h"

#include <xsec/enc/XSECCryptoException.hpp>
#include <xsec/framework/XSECException.hpp>

#include <ctime>
using namespace saml;
using namespace std;


SAMLResponse::SAMLResponse(
    const XMLCh* inResponseTo,
    const XMLCh* recipient,
    const Iterator<SAMLAssertion*>& assertions,
    SAMLException* e,
    const XMLCh* id,
    const SAMLDateTime* issueInstant
    ) : m_inResponseTo(XML::assign(inResponseTo)), m_issueInstant(NULL), m_recipient(XML::assign(recipient)), m_exception(NULL)
{
    RTTI(SAMLResponse);
    
    m_minor=SAMLConfig::getConfig().compatibility_mode ? 0 : 1;
    m_id=XML::assign(id);
    if (issueInstant) {
        m_issueInstant=new SAMLDateTime(*issueInstant);
        m_issueInstant->parseDateTime();
    }

    if (e) {
        e->setParent(this);
        m_exception=e;
        return;
    }
    
    while (assertions.hasNext())
        m_assertions.push_back(static_cast<SAMLAssertion*>(assertions.next()->setParent(this)));
}

SAMLResponse::SAMLResponse(DOMElement* e)
    : m_inResponseTo(NULL), m_issueInstant(NULL), m_recipient(NULL), m_exception(NULL)
{
    RTTI(SAMLResponse);
    fromDOM(e);
}

SAMLResponse::SAMLResponse(istream& in)
    : SAMLSignedObject(in), m_inResponseTo(NULL), m_issueInstant(NULL), m_recipient(NULL), m_exception(NULL)
{
    RTTI(SAMLResponse);
    fromDOM(m_document->getDocumentElement());
}

SAMLResponse::SAMLResponse(istream& in, int minor)
    : SAMLSignedObject(in,minor), m_inResponseTo(NULL), m_issueInstant(NULL), m_recipient(NULL), m_exception(NULL)
{
    RTTI(SAMLResponse);
    fromDOM(m_document->getDocumentElement());
}

SAMLResponse::~SAMLResponse()
{
    if (m_bOwnStrings) {
        XMLString::release(&m_inResponseTo);
        XMLString::release(&m_recipient);
    }
    delete m_issueInstant;
    for (vector<SAMLAssertion*>::const_iterator i=m_assertions.begin(); i!=m_assertions.end(); i++)
        delete (*i);
    delete m_exception;
}

void SAMLResponse::ownStrings()
{
    if (!m_bOwnStrings) {
        SAMLSignedObject::ownStrings();
        m_inResponseTo=XML::assign(m_inResponseTo);
        m_recipient=XML::assign(m_recipient);
        m_bOwnStrings = true;
    }
}

void SAMLResponse::fromDOM(DOMElement* e)
{
    SAMLObject::fromDOM(e);

    if (SAMLConfig::getConfig().strict_dom_checking && !XML::isElementNamed(e,XML::SAMLP_NS,L(Response)))
        throw MalformedException("SAMLResponse::fromDOM() requires samlp:Response at root");

    if (XMLString::parseInt(e->getAttributeNS(NULL,L(MajorVersion)))!=1)
        throw MalformedException(SAMLException::VERSIONMISMATCH,"SAMLResponse::fromDOM() detected incompatible response major version");

    m_minor=XMLString::parseInt(e->getAttributeNS(NULL,L(MinorVersion)));
    m_id=const_cast<XMLCh*>(e->getAttributeNS(NULL,L(ResponseID)));
    m_issueInstant=new SAMLDateTime(e->getAttributeNS(NULL,L(IssueInstant)));
    m_issueInstant->parseDateTime();
    if (e->hasAttributeNS(NULL,L(InResponseTo)))
        m_inResponseTo=const_cast<XMLCh*>(e->getAttributeNS(NULL,L(InResponseTo)));
    if (e->hasAttributeNS(NULL,L(Recipient)))
        m_recipient=const_cast<XMLCh*>(e->getAttributeNS(NULL,L(Recipient)));

    // Move to first child element.
    DOMElement* n=XML::getFirstChildElement(e);
    if (XML::isElementNamed(n,XML::XMLSIG_NS,L(Signature))) {
        SAMLInternalConfig& conf=dynamic_cast<SAMLInternalConfig&>(SAMLConfig::getConfig());
        try {
            m_signature=conf.m_xsec->newSignatureFromDOM(n->getOwnerDocument(),n);
            m_signature->load();
            m_sigElement=n;
        }
        catch(XSECException& e) {
            auto_ptr_char temp(e.getMsg());
            SAML_log.error("caught an XMLSec exception: %s",temp.get());
            throw MalformedException("caught an XMLSec exception while parsing signature: $1",params(1,temp.get()));
        }
        catch(XSECCryptoException& e) {
            SAML_log.error("caught an XMLSec crypto exception: %s",e.getMsg());
            throw MalformedException("caught an XMLSec crypto exception while parsing signature: $1",params(1,e.getMsg()));
        }

        // Move to Status element.
        n=XML::getNextSiblingElement(n);
    }

    auto_ptr<SAMLException> excep(SAMLException::getInstance(n));

    // If it's an error, toss it out.
    Iterator<saml::QName> codes=excep->getCodes();
    if (codes.hasNext()) {
        const saml::QName& code=codes.next();
        if (XMLString::compareString(XML::SAMLP_NS,code.getNamespaceURI()) || XMLString::compareString(L(Success),code.getLocalName()))
            excep->raise();
    }
    else
        excep->raise();

    n=XML::getNextSiblingElement(n,XML::SAML_NS,L(Assertion));
    while (n) {
        SAMLAssertion* a=new SAMLAssertion(n);
        m_assertions.push_back(static_cast<SAMLAssertion*>(a->setParent(this)));
        n=XML::getNextSiblingElement(n,XML::SAML_NS,L(Assertion));
    }
}

void SAMLResponse::setMinorVersion(int minor)
{
    m_minor=minor;
    ownStrings();
    setDirty();
}

void SAMLResponse::setInResponseTo(const XMLCh* inResponseTo)
{
    if (m_bOwnStrings)
        XMLString::release(&m_inResponseTo);
    else {
        m_inResponseTo=NULL;
        ownStrings();
    }
    m_inResponseTo=XML::assign(inResponseTo);
    setDirty();
}

void SAMLResponse::setIssueInstant(const SAMLDateTime* instant)
{
    delete m_issueInstant;
    m_issueInstant=NULL;
    if (instant) {
        m_issueInstant=new SAMLDateTime(*instant);
        m_issueInstant->parseDateTime();
    }
    ownStrings();
    setDirty();
}

void SAMLResponse::setRecipient(const XMLCh* recipient)
{
    if (m_bOwnStrings)
        XMLString::release(&m_recipient);
    else {
        m_recipient=NULL;
        ownStrings();
    }
    m_recipient=XML::assign(recipient);
    setDirty();
}

void SAMLResponse::setStatus(SAMLException* e)
{
    delete m_exception;
    m_exception=NULL;
    if (e)
        m_exception=static_cast<SAMLException*>(e->setParent(this));
    ownStrings();
    setDirty();
}

void SAMLResponse::setAssertions(const Iterator<SAMLAssertion*>& assertions)
{
    while (m_assertions.size())
        removeAssertion(0);
    while (assertions.hasNext())
        addAssertion(assertions.next());
}

void SAMLResponse::addAssertion(SAMLAssertion* assertion)
{
    if (assertion) {
        m_assertions.push_back(static_cast<SAMLAssertion*>(assertion->setParent(this)));
        ownStrings();
        setDirty();
    }
    else
        throw MalformedException("assertion cannot be null");
}

void SAMLResponse::removeAssertion(unsigned long index)
{
    SAMLAssertion* kill=m_assertions[index];
    m_assertions.erase(m_assertions.begin()+index);
    delete kill;
    ownStrings();
    setDirty();
}

void SAMLResponse::insertSignature()
{
    m_root->insertBefore(getSignatureElement(),m_root->getFirstChild());
}

DOMElement* SAMLResponse::buildRoot(DOMDocument* doc, bool xmlns) const
{
    DOMElement* r = doc->createElementNS(XML::SAMLP_NS, L(Response));
    r->setAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,samlp),XML::SAMLP_NS);
    r->setAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,saml),XML::SAML_NS);
    if (xmlns) {
        r->setAttributeNS(XML::XMLNS_NS,L(xmlns),XML::SAMLP_NS);
        r->setAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,xsi),XML::XSI_NS);
        r->setAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,xsd),XML::XSD_NS);
    }
    return r;
}

DOMNode* SAMLResponse::toDOM(DOMDocument* doc, bool xmlns) const
{
    SAMLObject::toDOM(doc,xmlns);
    DOMElement* r=static_cast<DOMElement*>(m_root);
    doc=r->getOwnerDocument();

    if (m_bDirty) {
        static const XMLCh One[]={chDigit_1, chNull};
        static const XMLCh Zero[]={chDigit_0, chNull};
        r->setAttributeNS(NULL,L(MajorVersion),One);
        r->setAttributeNS(NULL,L(MinorVersion),m_minor==0 ? Zero : One);

        // Only generate a new ID if we don't have one already.
        if (!m_id) {
            SAMLIdentifier id;
            m_id=XML::assign(id);
        }
        r->setAttributeNS(NULL,L(ResponseID),m_id);
        if (m_minor==1)
            r->setIdAttributeNS(NULL,L(ResponseID));

        if (!m_issueInstant) {
            m_issueInstant=new SAMLDateTime(time(NULL));
            m_issueInstant->parseDateTime();
        }
        r->setAttributeNS(NULL,L(IssueInstant),m_issueInstant->getRawData());

        if (!XML::isEmpty(m_inResponseTo))
            r->setAttributeNS(NULL,L(InResponseTo),m_inResponseTo);
        if (!XML::isEmpty(m_recipient))
            r->setAttributeNS(NULL,L(Recipient),m_recipient);

        if (m_exception)
            r->appendChild(m_exception->toDOM(doc,false));
        else {
            DOMElement* status=doc->createElementNS(XML::SAMLP_NS,L(Status));
            r->appendChild(status);
            DOMElement* code=doc->createElementNS(XML::SAMLP_NS,L(StatusCode));
            static const XMLCh samlp_Success[]=
            { chLatin_s, chLatin_a, chLatin_m, chLatin_l, chLatin_p, chColon,
              chLatin_S, chLatin_u, chLatin_c, chLatin_c, chLatin_e, chLatin_s, chLatin_s, chNull };
            code->setAttributeNS(NULL,L(Value),samlp_Success);
            status->appendChild(code);
        }
    
        for (vector<SAMLAssertion*>::const_iterator i=m_assertions.begin(); i!=m_assertions.end(); i++)
            r->appendChild((*i)->toDOM(doc));
        
        setClean();
    }
    else if (xmlns) {
        DECLARE_DEF_NAMESPACE(r,XML::SAMLP_NS);
        DECLARE_NAMESPACE(r,saml,XML::SAML_NS);
        DECLARE_NAMESPACE(r,samlp,XML::SAMLP_NS);
        DECLARE_NAMESPACE(r,xsi,XML::XSI_NS);
        DECLARE_NAMESPACE(r,xsd,XML::XSD_NS);
    }
    
    return m_root;
}

SAMLObject* SAMLResponse::clone() const
{
    SAMLResponse* r = new SAMLResponse(
        m_inResponseTo,
        m_recipient,
        getAssertions().clone(),
        m_exception ? static_cast<SAMLException*>(m_exception->clone()) : NULL,
        m_id,
        m_issueInstant
        );
    r->setMinorVersion(m_minor);
    return r;
}
