/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* SAMLObject.cpp - base class for all SAML constructs

   Scott Cantor
   2/21/02

   $History:$
*/

#include <xercesc/util/Base64.hpp>
#include <xercesc/util/XMLUTF8Transcoder.hpp>
#include <xercesc/framework/MemBufFormatTarget.hpp>
//#include <xsec/utils/XSECNameSpaceExpander.hpp>
//#include <xsec/canon/XSECC14n20010315.hpp>

#include "internal.h"
using namespace saml;

#include <sstream>
using namespace std;

SAMLObject::SAMLObject() : m_root(NULL), m_document(NULL), m_parent(NULL), m_bDirty(true), m_bOwnStrings(true), m_log(NULL)
{
    RTTI(SAMLObject);
}

SAMLObject::SAMLObject(istream& in) : m_root(NULL), m_document(NULL), m_parent(NULL), m_bDirty(true), m_bOwnStrings(true), m_log(NULL)
{
    RTTI(SAMLObject);
    try {
        XML::Parser p;
        XML::StreamInputSource src(in);
        Wrapper4InputSource dsrc(&src,false);
        m_document=p.parse(dsrc);
    }
    catch (XMLException& e) {
        auto_ptr_char buf(e.getMessage());
        SAML_log.error("caught an XML exception while parsing an istream&:\n"
                     "\tFile: %s\tLine: %u\n\tMessage: %s",
                     e.getSrcFile(),e.getSrcLine(),buf.get());
        throw MalformedException(string("SAMLObject() caught XML exception: ") + buf.get());
    }
}

SAMLObject::SAMLObject(istream& in, int minor)
    : m_root(NULL), m_document(NULL), m_parent(NULL), m_bDirty(true), m_bOwnStrings(true), m_log(NULL)
{
    RTTI(SAMLObject);
    try {
        XML::Parser p(minor);
        XML::StreamInputSource src(in);
        Wrapper4InputSource dsrc(&src,false);
        m_document=p.parse(dsrc);
    }
    catch (XMLException& e) {
        auto_ptr_char buf(e.getMessage());
        SAML_log.error("caught an XML exception while parsing an istream&:\n"
                     "\tFile: %s\tLine: %u\n\tMessage: %s",
                     e.getSrcFile(),e.getSrcLine(),buf.get());
        throw MalformedException(string("SAMLObject() caught XML exception: ") + buf.get());
    }
}

SAMLObject::~SAMLObject()
{
    if (m_document)
        m_document->release();
}

void SAMLObject::_RTTI(const char* classname)
{
    m_classname=classname;
    m_log=&logging::Category::getInstance(string(SAML_LOGCAT) + '.' + m_classname);
}

SAMLObject* SAMLObject::setParent(SAMLObject* parent)
{
    if (m_parent)
        throw SAMLException("SAMLObject::setParent() called on an already-contained object");
    if (!parent)
        throw SAMLException("SAMLObject::setParent() called with null parameter");
    m_parent = parent;
    return this;
}

void SAMLObject::setDirty()
{
    m_bDirty=true;
    if (m_parent)
        m_parent->setDirty();
}

DOMNode* SAMLObject::toDOM(DOMDocument* doc, bool xmlns) const
{
    checkValidity();
    
    // Make sure we have a document to use.
    if (!doc) {
        if (m_root)
            doc=m_root->getOwnerDocument();     // favor an existing DOM
        else if (m_document)
            doc=m_document;                     // partial success earlier?
        else {                                  // go ahead and build a new one
            DOMImplementation* impl=DOMImplementationRegistry::getDOMImplementation(NULL);
            m_document=doc=impl->createDocument();
        }
    }
    
    // Are we built?
    if (m_root) {
        // Can we use what we have?
        if (!m_bDirty) {
            // If the DOM tree is already generated, compare the Documents.
            if (m_root->getOwnerDocument() != doc) {
                // We already built a tree. Just adopt it into the new document.
                // XXX: this won't actually work, Xerces-C doesn't support adoptNode
                // But there's no way to safely import because child objects won't get
                // notified and be able to update their roots.
                m_root = doc->adoptNode(m_root);
                
                // Release anything we have now.
                if (m_document) {
                    m_document->release();
                    m_document = NULL;
                }
            }
            return m_root;
        }
    }
    
    // This means we needed a new DOM, for whatever reason. We start by
    // letting each subtype construct its own root element.
    // The key here for dirty DOMs is to control Node memory mgmt sole via
    // the document owner. We count on the rest to be garbage collected
    // at that point, so if we end up orphaning part of a DOM we built
    // earlier, it should get cleaned up once the document is released.
    return m_root = buildRoot(doc,xmlns);
}

void SAMLObject::fromDOM(DOMElement* e)
{
    if (!e)
        throw MalformedException("SAMLObject::fromDOM() given an empty DOM");
    m_root = e;
    m_bOwnStrings = false;
    setClean();    // we have to be clean if we're starting from existing XML
}

DOMNode* SAMLObject::plantRoot()
{
    if (m_root)
    {
        DOMNode* domroot=m_root;
        while (domroot->getParentNode() && domroot->getParentNode()->getNodeType()!=DOMNode::DOCUMENT_NODE)
            domroot=domroot->getParentNode();

        DOMElement* e=m_root->getOwnerDocument()->getDocumentElement();
        if (e && e!=domroot)
            m_root->getOwnerDocument()->replaceChild(domroot,e);
        else if (!e)
            m_root->getOwnerDocument()->appendChild(domroot);
    }
    return m_root;
}

XMLByte* SAMLObject::toBase64(unsigned int* outputLength) const
{
    ostringstream os;
    os << *this;
    string osstr=os.str();
    return Base64::encode(reinterpret_cast<const XMLByte*>(osstr.data()),osstr.size(),outputLength);
}

namespace {
    static XMLCh UTF8[]={ chLatin_U, chLatin_T, chLatin_F, chDigit_8, chNull };
}

ostream& saml::xmlout(ostream& target, const XMLCh* s)
{
    if (!s)
        return target;
    char* p=toUTF8(s);
    target << p;
    delete[] p;
    return target;
}

ostream& saml::operator<<(ostream& ostr, const DOMNode& node)
{
    static const XMLCh impltype[] = { chLatin_L, chLatin_S, chNull };
    DOMImplementation* impl=DOMImplementationRegistry::getDOMImplementation(impltype);
    DOMWriter* serializer=(static_cast<DOMImplementationLS*>(impl))->createDOMWriter();
    serializer->setEncoding(UTF8);
    try
    {
        MemBufFormatTarget target;
        if (!serializer->writeNode(&target,node))
            throw SAMLException("operator <<: unable to serialize XML instance");
        unsigned int len=target.getLen();
        const XMLByte* byteptr=target.getRawBuffer();
        while (len--)
            ostr << (char)*(byteptr++);
    }
    catch (...)
    {
        serializer->release();
        throw;
    }
    serializer->release();
    return ostr;
}

ostream& saml::operator<<(ostream& ostr, const SAMLObject& obj)
{
    DOMNode* n=obj.toDOM();
    if (n)
        ostr << *n;
    else
        ostr << "{null SAML object}" << endl;
    return ostr;
}

char* saml::toUTF8(const XMLCh* src)
{
    unsigned int eaten;
    unsigned int srclen=XMLString::stringLen(src);
    XMLUTF8Transcoder t(UTF8, srclen*4 + 1);
    char* buf=new char[srclen*4 + 1];
    memset(buf,0,srclen*4 + 1);
    t.transcodeTo(
        src,srclen,
        reinterpret_cast<XMLByte*>(buf),srclen*4,
        eaten,XMLTranscoder::UnRep_RepChar);
    return buf;
}

XMLCh* saml::fromUTF8(const char* src)
{
    unsigned int eaten;
    unsigned int srclen=strlen(src);
    XMLUTF8Transcoder t(UTF8, srclen + 1);
    XMLCh* buf=new XMLCh[srclen + 1];
    unsigned char* sizes=new unsigned char[srclen];
    memset(buf,0,(srclen+1)*sizeof(XMLCh));
    t.transcodeFrom(
        reinterpret_cast<const XMLByte*>(src),srclen,
        buf,srclen,
        eaten,sizes);
    delete[] sizes;
    return buf;
}
