#!/usr/bin/perl -w

# installtool.pl - Install script for NagiosGrapher
# Copyright (C) 2004, NETWAYS GmbH, Gerd Mueller, Marius Hein
# $Id: install.pl 1036 2005-12-01 16:33:03Z mhein $
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

use lib '../';
use strict;

use subs qw(
	print_welcome
	print_end
	print_n1
	print_n2
	print_net
	print_pipe
	print_summary
	get_reply
	get_yn
	get_value
	get_srvext_type
	get_default
	validate_dir
	validate_file
	validate_user
	validate_group
	validate_loglevel
	detect_nagios
	detect_nagios_version
	init_flavors
	questions_default
	questions_service_ext
	questions_pipe
	remove_config_questions
	remove_all_values
	install_config_files
	install_cgi_files
	install_daemon_files
	install_perl_files
	install_image_files
	install_rc_files
	alter_config_files
	alter_module_file
	alter_rc_script
	rl
	ng_user
	ng_group
	ng_daemon
);

use Getopt::Long;
use NagiosGrapher;
use POSIX;
use Term::UI;
use Term::ReadLine;
use Data::Dumper;
use File::Copy;
use File::Basename;

my (%steps, %defaults, $nagios, $nagios_version, $flavor, %flavors);

%flavors = (
	'default' => 1,
	'debian' => 2
);

my $term = new Term::ReadLine('NagiosGrapherInstallation');

my ($update, $install);

GetOptions (
	'install'	=> \$install,
	'i'			=> \$install,

	'update'	=> \$update,
	'u'			=> \$update,

	'flavor=s'	=> \$flavor,
	'f=s'		=> \$flavor,
);

if (!$update && !$install || $update && $install) {
	print "Please use --update or --install for operations!\n\n";
	exit (1);
}

# stopping the daemon if running ...
ng_daemon 'stop';

RESTART_ALL:

# detecting nagios, version and the flavor

if ($flavor && !exists $flavors{$flavor}) {
	printf("Not a valid flavor '%s'. Use %s.\n", $flavor, join(', ', keys(%flavors)));
	exit (0);
}

($nagios, my $t_flavor) = detect_nagios();
$flavor = $t_flavor unless($flavor);
$nagios_version = detect_nagios_version();

# init the install flavors ...
init_flavors();

# Loading all questions ...
questions_default();

# Remove the config questions ...
if ($update) {
	remove_config_questions();
}

# Asking some boring questions ...
if (print_welcome()) {
	RESTART:
	foreach (sort(keys(%steps))) {
		my $step = $steps{$_};
		if (exists($step->{value}) && !defined($step->{value})) {
			if (exists($step->{'choice'}) && defined($step->{'choice'})) {
				$steps{$_}{'value'} = $step->{'type'}
				(
					prompt => $step->{'question'},
					default => $step->{'default'},
					allow => $step->{'validate'},
					choices => $step->{'choice'},
				);
			}
			else {
				$steps{$_}{'value'} = $step->{'type'}
				(
					prompt => $step->{'question'},
					default => $step->{'default'},
					allow => $step->{'validate'},
				);
			}

			print "\n\n";

			# Depending decisions ...
			# ~~~~~~~~~~~~~~~~~~~~~~~
			#
			# NG_SRVEXT => The serviceextquestion ...
			if ($_ eq 'NG_SRVEXT') {
				questions_service_ext ($step->{value});
				goto RESTART;
			}
			# NG_INTERFACE => Pipelocation
			elsif ($_ eq 'NG_INTERFACE') {
				questions_pipe ($step->{value});
				goto RESTART;
			}
		}
	}

	print_summary();

	# go to the files we want
	chdir '../';

	# install the config files
	install_config_files();

	# alter the config file
	alter_config_files();

	# install the cgi files
	install_cgi_files();

	# install the images
	install_image_files();

	# install the daemon files and tools
	install_daemon_files();

	# installing the perl module(s)
	install_perl_files();

	# alter the perlmodule
	alter_module_file();

	# and now the startscript
	install_rc_files();

	# alter the RC script
	alter_rc_script();

	# and starting the daemon
	ng_daemon 'start';

	# hhh (..) and to the end ...
	print_end();

	exit (0);
}
else {
	print "Abort !\n";
	exit (1);
}

exit (0);

sub print_summary {
	system 'clear';
	print "---\nYour answers:\n";
	print $term->history_as_string();
	print "\n---\n\n";

	sleep 1;
	my $re = get_yn(prompt => 'Are u ready to install ?', default => 'y');

	unless ($re) {
		remove_all_values();
		print "\n";
		goto RESTART_ALL;
	}

	system 'clear';
}

sub print_welcome {
	print "Welcome to the NagiosGrapher". $NagiosGrapher::VERSION. " install tool.\n\n";

	# my $re = $term->ask_yn(
	# 	prompt => 'Are you ready ?',
	# 	default => 'n',
	# );

	print "\n";

	# return $re;

	return 1;
}

sub print_end {
	print "\n";
	unless ($update) {
		print
			"Installation finished successfully. Now you have to do some\n",
			"manually work to bring up the grapher.\n\n",
			"First you have to enable the process-service-perfdata within Nagios:\n\n",
			"\t- Set process_performance_data=1 in your nagios.cfg\n",
			"\t- Define a service_perfdata_command in the nagios.cfg\n\t(e.g. service_perfdata_command=process-service-perfdata)\n\n",
			"\t- Write the following command in misccommands.cfg:\n\n";

		if (get_value('NG_INTERFACE') eq 'pipe') {
			print_pipe();
		}
		elsif (get_value('NG_INTERFACE') eq 'network') {
			print_net();
		}

		if (get_value('NAGIOS_VERSION') eq '1.x') {
			print_n1();
		}
		elsif (get_value('NAGIOS_VERSION') eq '2.x') {
			print_n2();
		}
	}
	# ---
	# Don't remove this lines below, because of the GPLv2 licence ...
	# Thanks.
	# ---
	print
		"So good luck for this !\n",
		"For further information goto http://www.nagiosexchange.org/NagiosGrapher.84.0.html\n",
		"There's a lot of stuff you can need for the effort of configuring :-)\n\n",
		"(c) 2005 NETWAYS GmbH <www.netways.de>\n";


}

sub print_n1 {
	print
		"\tFor Nagios 1.x you have to define the xed_templatefile for serviceextinfo\n",
		"\tin the cgi.cfg:\n",
		"\t- xedtemplate_config_file=". get_value('NG_SRVEXT_FILE'). "\n\n";
}

sub print_n2 {
	print
		"\tFor Nagios 2.x you have to include the serviceextinfo\n",
		"\tin the nagios.cfg:\n\n",
		"\tFor Type MULTIPLE:\n",
		"\t- cfg_dir=". get_value('NG_SRVEXT_DIR'). "\n\n",
		"\tFor Type SINGLE:\n",
		"\t- cfg_file=". get_value('NG_SRVEXT_FILE'). "\n\n",
		"\tDon't forget to reload if a serviceext entry is written !!!\n\n";
}

sub print_net {
	print
	"\tdefine command{\n",
    "\t\tcommand_name    process-service-perfdata\n",
    "\t\tcommand_line    echo '\$HOSTNAME\$\\t\$SERVICEDESC\$\\t\$OUTPUT\$\\t\$PERFDATA\$\\n' | nc -u -w 3 localhost 5667\n",
    "\t}\n\n",
    "or you try the udpecho.c tool which doesn't wait for and answer ...\n\n";
}

sub print_pipe {
	print
	"\tdefine command{\n",
	"\t\tcommand_name    process-service-perfdata\n",
	"\t\tcommand_line    ". get_value('NAGIOS_CONTRIBUTION'). "/fifo_write.pl ". get_value('NG_INTERFACE_PIPE'). " \"\$HOSTNAME\$\\t\$SERVICEDESC\$\\t\$OUTPUT\$\\t\$PERFDATA\$\n\" 3\n",
	"\t}\n\n";
}

sub questions_default {
	%steps = (

		NAGIOS_VERSION => {
			type => \&get_reply,
			validate => \&validate_blank,
			choice => ['1.x', '2.x'],
			question => 'Which is you Nagios version ?',
			default => '1.x',
			value => get_default('NAGIOS_VERSION'),
		},

		NAGIOS_CONFIG => {
			type => \&get_reply,
			validate => \&validate_file,
			choice => undef,
			question => 'Where is your Nagios Config ?',
			default => get_default('NAGIOS_CONFIG'),
			value => undef,
		},

		NAGIOS_CONFIG_CGI => {
			type => \&get_reply,
			validate => \&validate_file,
			choice => undef,
			question => 'Where is your Nagios CGI Config ?',
			default => get_default('NAGIOS_CONFIG_CGI'),
			value => undef,
		},

		NAGIOS_IMAGES => {
			type => \&get_reply,
			validate => \&validate_dir,
			choice => undef,
			question => 'Where is your Nagios images folder ?',
			default => get_default('NAGIOS_IMAGES'),
			value => undef,
		},

		NAGIOS_IMAGES_LOGOS => {
			type => \&get_reply,
			validate => \&validate_dir,
			choice => undef,
			question => 'Where is your Nagios images/logos folder ?',
			default => get_default('NAGIOS_IMAGES_LOGOS'),
			value => undef,
		},

		NAGIOS_FOLDER_CGI => {
			type => \&get_reply,
			validate => \&validate_dir,
			choice => undef,
			question => 'Where is your Nagios cgi-bin folder ?',
			default => get_default('NAGIOS_FOLDER_CGI'),
			value => undef,
		},

		NAGIOS_PROC_USER => {
			type => \&get_reply,
			validate => \&validate_user,
			choice => undef,
			question => 'The Nagios process user ?',
			default => get_default('NAGIOS_PROC_USER'),
			value => undef,
		},

		NAGIOS_PROC_GROUP => {
			type => \&get_reply,
			validate => \&validate_group,
			choice => undef,
			question => 'The Nagios process group ?',
			default => get_default('NAGIOS_PROC_GROUP'),
			value => undef,
		},

		APACHE_USER => {
			type => \&get_reply,
			validate => \&validate_user,
			choice => undef,
			question => 'The Apache process owner ?',
			default => get_default('APACHE_USER'),
			value => undef,
		},

		PERL_INC => {
			type => \&get_reply,
			validate => \&validate_dir,
			choice => undef,
			question => 'Where is your perl/inc folder ?',
			default => get_default('PERL_INC'),
			value => undef,
		},

		NAGIOS_CONTRIBUTION => {
			type => \&get_reply,
			validate => \&validate_dir,
			choice => undef,
			question => 'The folder for Nagios contributions (e.g. collect2 and fifo_write) ?',
			default => get_default('NAGIOS_CONTRIBUTION'),
			value => undef,
		},

		NG_CONFIG => {
			type => \&get_reply,
			validate => \&validate_dir,
			choice => undef,
			question => 'The folder for the grapher configuration ?',
			default => get_default('NG_CONFIG'),
			value => undef,
		},

		NG_DAEMON => {
			type => \&get_reply,
			validate => \&validate_blank,
			choice => undef,
			question => 'The folder for daemon data  ?',
			default => get_default('NG_DAEMON'),
			value => undef,
		},

		NG_INTERFACE => {
			type => \&get_reply,
			validate => undef,
			choice => ['pipe', 'network'],
			question => 'Which datainterface you\'ll use  ?',
			default => get_default('NG_INTERFACE'),
			value => undef,
		},

		NG_SRVEXT => {
			type => \&get_reply,
			validate => undef,
			choice => ['MULTIPLE', 'SINGLE'],
			question => 'Which type of service ext you\'ll use  ?',
			default => get_default('NG_SRVEXT'),
			value => undef,
		},

		NG_LOGFILE => {
			type => \&get_reply,
			validate => \&validate_blank,
			choice => undef,
			question => 'Where should I store the logfile ?',
			default => get_default('NG_LOGFILE'),
			value => undef,
		},

		NG_LOGLEVEL => {
			type => \&get_reply,
			validate => \&validate_loglevel,
			choice => undef,
			question => 'Which loglevel should i use ?',
			default => get_default('NG_LOGLEVEL'),
			value => undef,
		},

		NG_RRD => {
			type => \&get_reply,
			validate => \&validate_blank,
			choice => undef,
			question => 'Where should I store the rrd\'s ?',
			default => get_default('NG_RRD'),
			value => undef,
		},
	);
}

sub detect_nagios {
	my $nagios = undef;
	my $flavor = undef;
	my %search = (
		'debian' => '/usr/sbin/nagios',
		'default' => '/usr/local/nagios/bin/nagios'
	);


	foreach (keys(%search)) {
		$nagios = $search{$_} if ( -x $search{$_} );
		$flavor = $_

	}

	unless ($nagios) {
		$nagios = `which nagios`;
		$flavor = 'debian'
	}

	chomp $nagios;
	chomp $flavor;
	return ($nagios, $flavor);
}

sub detect_nagios_version {
	if ($nagios && -x $nagios) {
		my $check = `$nagios -V`;
		if ($check =~ /^Nagios\s+([0-9,.]+)/mi) {
			if ($1 >= 3) {
				$check = '3.x';
			}
			elsif ($1 >= 2) {
				$check = '2.x';
			}
			elsif ($1 > 0) {
				$check = '1.x';
			}

			return $check;
		}
	}

	return undef;
}

sub get_srvext_type {
	my $type = 'SINGLE';
	if ($nagios_version eq '2.x') {
		$type = 'MULTIPLE';
	}
	return $type;
}

sub get_default {
	my ($key) = @_;

	if ($key && exists $defaults{$flavor}->{$key}) {
		return $defaults{$flavor}->{$key};
	}

	return undef;
}

sub init_flavors {
	%defaults = (
		'debian' => {
			NAGIOS_VERSION => $nagios_version,
			NAGIOS_CONFIG => '/etc/nagios/nagios.cfg',
			NAGIOS_CONFIG_CGI => '/etc/nagios/cgi.cfg',
			NAGIOS_IMAGES => '/usr/share/nagios/htdocs/images',
			NAGIOS_IMAGES_LOGOS => '/usr/share/nagios/htdocs/images/logos',
			NAGIOS_FOLDER_CGI => '/usr/lib/cgi-bin/nagios',
			NAGIOS_PROC_USER => 'nagios',
			NAGIOS_PROC_GROUP => 'nagios',
			APACHE_USER => 'www-data',
			PERL_INC => '/etc/perl',
			NAGIOS_CONTRIBUTION => '/usr/share/nagios/contrib',
			NG_CONFIG => '/etc/nagios',
			NG_DAEMON => '/var/run/nagios_grapher',
			NG_INTERFACE => 'pipe',
			NG_SRVEXT => get_srvext_type,
			NG_LOGFILE => '/var/log/nagios/ngraph.log',
			NG_LOGLEVEL => '255',
			NG_RRD => '/var/lib/nagios/rrd',
			NG_INTERFACE_PIPE => '/var/lib/nagios/ngraph.pipe',
			NG_SRVEXT_FILE => '/etc/nagios/serviceext.cfg',
			NG_SRVEXT_DIR => '/etc/nagios/serviceext',

		},

		'default' => {
			NAGIOS_VERSION => $nagios_version,
			NAGIOS_CONFIG => '/usr/local/nagios/etc/nagios.cfg',
			NAGIOS_CONFIG_CGI => '/usr/local/nagios/etc/cgi.cfg',
			NAGIOS_IMAGES => '/usr/local/nagios/share/images',
			NAGIOS_IMAGES_LOGOS => '/usr/local/nagios/share/images/logos',
			NAGIOS_FOLDER_CGI => '/usr/local/nagios/sbin',
			NAGIOS_PROC_USER => 'nagios',
			NAGIOS_PROC_GROUP => 'nagios',
			APACHE_USER => 'www-data',
			PERL_INC => '/etc/perl',
			NAGIOS_CONTRIBUTION => '/usr/share/nagios/contrib',
			NG_CONFIG => '/usr/local/nagios/etc',
			NG_DAEMON => '/usr/local/nagios/var/nagios_grapher',
			NG_INTERFACE => 'network',
			NG_SRVEXT => get_srvext_type,
			NG_LOGFILE => '/usr/local/nagios/var/ngraph.log',
			NG_LOGLEVEL => '255',
			NG_RRD => '/usr/local/nagios/var/rrd',
			NG_INTERFACE_PIPE => '/usr/local/nagios/var/rw/ngraph.pipe',
			NG_SRVEXT_FILE => '/usr/local/nagios/etc/serviceext.cfg',
			NG_SRVEXT_DIR => '/usr/local/nagios/etc/serviceext',
		}
	)
}

sub remove_config_questions {
	my @remove = qw(
		NAGIOS_VERSION
		NAGIOS_CONFIG
		NAGIOS_CONFIG_CGI
		NG_INTERFACE
		NG_SRVEXT
		NG_LOGFILE
		NG_LOGLEVEL
		NG_RRD
	);

	foreach (@remove) {
		delete $steps{$_};
	}

	return 1;
}

sub remove_all_values {
	while (my($key, $value) = each(%steps)) {
		$value->{'value'} = undef;
	}
}

sub questions_pipe {
	my ($type) = @_;
	if ($type eq 'pipe') {
		$steps{'NG_INTERFACE_PIPE'} = {
			type => \&get_reply,
			validate => \&validate_blank,
			choice => undef,
			question => 'Where should i store the pipe  ?',
			default => get_default('NG_INTERFACE_PIPE'),
			value => undef,
		};
		return 1;
	}

	return;
}

sub questions_service_ext {
	my ($type) = @_;

	if ($type eq 'SINGLE') {
		$steps{'NG_SRVEXT_FILE'} = {
			type => \&get_reply,
			validate => \&validate_blank,
			choice => undef,
			question => 'Your Serviceextfile  ?',
			default => get_default('NG_SRVEXT_FILE'),
			value => undef,
		};
	}
	elsif ($type eq 'MULTIPLE') {
		$steps{'NG_SRVEXT_DIR'} = {
			type => \&get_reply,
			validate => \&validate_blank,
			choice => undef,
			question => 'Your Serviceext directory  ?',
			default => get_default('NG_SRVEXT_DIR'),
			value => undef,
		};
	}

	return 1;
}

sub install_rc_files {

	my $name = 'update-rc.d';
	my $cmd = `which $name`;
	chomp $cmd;

	my $dir = '/etc/init.d';
	my $file = 'nagios_grapher';

	print "Installing init script and links ";

	if (!($update && -e $dir . '/'. $file)) {
		if (-e $dir. '/'. $file) {
			unlink $dir. '/'. $file;
		}

		system('cp', '-f', $file, $dir);
	}

	if ($cmd && -x $cmd) {
		system('update-rc.d', '-f', $file, 'remove');
		system('update-rc.d', $file, 'defaults');
	}
	else {
		print "debian like command '$name' not found !\n";
	}

	print "...done.\n";

	return 0;
}

sub install_daemon_files {
	my $dir = get_value 'NAGIOS_CONTRIBUTION';
	my @files = qw(
		collect2.pl
		fifo_write.pl
		contrib/fifo_write/C/fifo_write.c
		contrib/fifo_write/tcp/fifo_write_from_tcp.pl
	);

	print "Installing daemon files";

	while (my $file = shift @files) {
		my $dest_file = $dir.  '/'. basename($file);

		#unless ($dest_file =~ m/$dir/i) {

			if ( -e $dest_file) {
				unlink $dest_file;
			}

			system('cp', '-f', $file, $dest_file);
			system('chown', ng_user. ':'. ng_group, $dest_file);
			system('chmod', '0700', $dest_file);

			print ".";
		#}
	}

	# Compiling the fifo_write.c file ...
	if (!$update && get_yn(prompt => "\n". 'Should I compile fifo_write.c for you ?', default => 'n')) {
		my $gcc = `which gcc`;
		chomp $gcc;

		if ( -x $gcc ) {
			print "Compiling fifo_write.c ";

			if ( -e $dir. '/'. 'fifo_write') {
				unlink $dir. '/'. 'fifo_write';
			}

			system($gcc, '-o', $dir. '/'. 'fifo_write', $dir. '/'. 'fifo_write.c');
			system('chown', ng_user. ':'. ng_group, $dir. '/'. 'fifo_write');
			system('chmod', '0700', $dir. '/'. 'fifo_write');

			print "...done.\n\n";
		}
	}

	print "done.\n";

	return 1;
}

sub install_perl_files {
	my $dir = get_value 'PERL_INC';
	my @files = qw(NagiosGrapher.pm);

	print "Installing PERL modules";
	while (my $file = shift @files) {
		if (-e $dir. '/'. $file) {
			unlink $dir. '/'. $file;
		}
		system('cp', '-f', $file, $dir);
		print ".";
	}
	print "done.\n";

	return 1;
}

sub install_image_files {
	my $dir = get_value 'NAGIOS_IMAGES_LOGOS';
	my @files = qw(dot.png graph.png);

	print "Installing images";
	while (my $file = shift @files) {
		if (-e $dir. '/'. $file) {
			unlink $dir. '/'. $file;
		}

		system('cp', '-f', $file, $dir);
		system('chown', ng_user. ':'. get_value('APACHE_USER'), $dir. '/'. $file);
		system('chmod', 'g+r', $dir. '/'. $file);

		print ".";
	}
	print "done.\n";

	return 1;
}

sub install_cgi_files {
	my $dir = get_value 'NAGIOS_FOLDER_CGI';
	my @files = qw(graphs.cgi rrd2-graph.cgi rrd2-system.cgi);

	print "Installing CGI files";

	while (my $file = shift @files) {
		if (-e $dir. '/'. $file) {
			unlink $dir. '/'. $file;
		}

		system('cp', '-f', $file, $dir);
		system('chown', ng_user. ':'. get_value('APACHE_USER'), $dir. '/'. $file);
		system('chmod', 'u+x', $dir. '/'. $file);
		system('chmod', 'g+x', $dir. '/'. $file);
	}

	print "...done.\n";

	return 1;
}

sub install_config_files {
	return 1 if ($update);

	my $dir = $steps{'NG_CONFIG'}->{'value'};

	print "Installing configfiles";

	POSIX::mkdir($dir. '/ngraph.d', 755);

	if (-e $dir. '/ngraph.d') {
		system('rm', '-rf', $dir. '/ngraph.d');
	}

	if (-e $dir. '/ngraph.ncfg') {
		unlink $dir. '/ngraph.ncfg';
	}

	system('cp', '-rf', 'cfg/templates/', $dir. '/ngraph.d/');
	system('cp', '-f', 'cfg/nmgraph.ncfg', $dir. '/ngraph.d/');
	system('cp', '-f', 'cfg/ngraph.ncfg', $dir);

	system('chown', '-R', ng_user. ':'. ng_group, $dir. '/ngraph.d/');
	system('chown', '-R', ng_user. ':'. ng_group, $dir. '/ngraph.ncfg');

	print "...done.\n";

	return 1;
}

sub alter_rc_script {
	return 1 if ($update);

	my $file = '/etc/init.d/nagios_grapher';
	if ( -e $file) {

		open RC, '<', $file;
		my @lines = <RC>;
		close RC;

		my %rc_map = (
			'^DAEMON=.*?\$NAME\.*?$' => 'DAEMON='. get_value('NAGIOS_CONTRIBUTION'). '/$NAME',
			'^PIDFILE=.*?\$DESC\.pid.*?$' => 'PIDFILE='. get_value('NG_DAEMON'). '/$DESC.pid',
		);

		print "Alter rcfile";
		open RC, '>', $file;
		foreach my $line (@lines) {
			while (my ($key, $val) = each(%rc_map)) {
				if ($line =~ m/$key/) {
					$line =~ s/$key/# INSTALLER:\n$val\n/;
					$line =~ s/\\//g;

					print ".";
				}
			}
			print RC $line;
		}
		close RC;
		print "done.\n";

		return 1;
	}

	return 0;

}

sub alter_module_file {
	# return 1 if ($update);
	my $file = get_value('PERL_INC') . '/NagiosGrapher.pm';
	if ( -e $file ) {
		print "Alter perlmodule";
		my %perl_map = (
			'our \$ngraphcfg' => get_value('NG_CONFIG'). '/ngraph.ncfg',
			'our \$object_cache' => get_value('NG_DAEMON'). '/NagiosGrapher.obj'
		);

		open PM, '<', $file;
		my @lines = <PM>;
		close PM;

		open PM, '>', $file;
		foreach my $line (@lines) {
			while (my($key, $val) = each(%perl_map)) {
				if ($line =~ m/^$key.*?$/) {
					$line =~ s/^$key.*?$/# INSTALLER:\n$key = '$val'\;\n/;
					$line =~ s/\\//g;
					print ".";
				}
			}

			print PM $line;
		}
		close PM;


		print "done.\n";
	}
}

sub alter_config_files {
	return 1 if ($update);


	my $file = my $dir = $steps{'NG_CONFIG'}->{'value'}. '/ngraph.ncfg';
	if (! -e $file) {
		die ("alter_config_files: $file not found !");
	}

	print "Alter the configfile";

	open CFG, '<', $file;
	my @lines = <CFG>;
	close CFG;

	my %conf_map = (
		'pipe'				=> 'NG_INTERFACE_PIPE',
		'interface'			=> 'NG_INTERFACE',
		'pidfile'			=> 'NG_DAEMON',
		'user'				=> 'NAGIOS_PROC_USER',
		'group'				=> 'NAGIOS_PROC_GROUP',
		'rrdpath'			=> 'NG_RRD',
		'nagios_config'		=> 'NAGIOS_CONFIG',
		'cgi_config'		=> 'NAGIOS_CONFIG_CGI',
		'log_file'			=> 'NG_LOGFILE',
		'log_level'			=> 'NG_LOGLEVEL',
		'cfg_dir='			=> 'NG_CONFIG',
		'serviceext_type'	=> 'NG_SRVEXT',
		'serviceextinfo'	=> 'NG_SRVEXT_FILE',
		'serviceext_path'	=> 'NG_SRVEXT_DIR'
	);

	my $value = undef;

	open CFG, '>', $file;
	while (my $line = shift @lines) {
		while (my ($config_key, $install_key) = each(%conf_map)) {
			if ($config_key eq 'pidfile') {
				$value = get_value($install_key). '/nagios_grapher.pid';
			}
			elsif ($config_key =~ m/.*?path.*?/i && get_value($install_key)) {
				$value = get_value($install_key). '/';
			}
			elsif ($config_key eq 'cfg_dir=') {
				$value = get_value($install_key). '/ngraph.d';
			}
			else {
				$value = get_value($install_key);
			}
			my $ol = $line;
			$line = rl($line, $config_key, $value);
			if ($line ne $ol) {
				print ".";
			}
		}

		print CFG $line;

	}
	close CFG;

	print "done.\n";

	return 1;
}

sub validate_blank {
	return 1;
}

sub validate_file {
	my ($file) = @_;
	if ( -e $file ) {
		return 1;
	}

	return;
}

sub validate_dir {
	my ($dir) = @_;
	if ( -d $dir ) {
		return 1;
	}

	return;
}

sub validate_user {
	my ($user) = @_;
	my @data = POSIX::getpwnam($user);
	if (scalar(@data) > 0) {
		return 1;
	}

	return;
}

sub validate_group {
	my ($group) = @_;
	my @data = POSIX::getgrnam($group);
	if (scalar(@data) > 0) {
		return 1;
	}

	return;
}

sub validate_loglevel {
	my ($level) = @_;
	if ($level > 0 && $level <= 255) {
		return 1
	}

	return
}

sub get_reply {
	return $term->get_reply(@_);
}

sub get_yn {
	return $term->ask_yn(@_);
}

sub get_value {
	my ($key) = @_;
	if (exists($steps{$key}) && defined($steps{$key}->{'value'})) {
		return $steps{$key}->{'value'};
	}

	return undef;
}

sub rl {
	my ($line, $key, $replace, $debug) = @_;

	if ($line =~ m/\s*$key/ && defined($replace) && length($replace) > 0) {
		if ($key =~ m/=/) {
			$line =~ s/$key.*?$/$key$replace\n/;
		}
		else {
			$line =~ s/$key.*?$/$key\t\t$replace\n/;
		}

		print $line if ($debug);
	}

	return $line;
}

sub ng_user {
	return get_value('NAGIOS_PROC_USER');
}

sub ng_group {
	return get_value('NAGIOS_PROC_GROUP');
}

sub ng_daemon {
	my $daemon = '/etc/init.d/nagios_grapher';
	my ($type) = @_;
	if (-x $daemon) {
		if ($type =~ m/start|stop/) {
			system($daemon, $type);

			return 1;
		}
	}

	return;
}

1;
# [EOF]
