/*
 *  inf.c
 *  mod_musicindex
 *
 *  $Id: sort.c 634 2006-06-26 20:23:42Z varenet $
 *
 *  Created by Thibaut VARENE on Thu Mar 20 2003.
 *  Copyright (c) 2003-2004 Regis BOUDIN
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version 2.1,
 *  as published by the Free Software Foundation.
 *
 */

/**
 * @file
 * Sorting routines.
 *
 * @author Regis Boudin
 * @version $Revision: 634 $
 * @date 2003-2004
 *
 * @todo Regis, please document (eventually).
 * @todo review int types
 */

#include "sort.h"

#ifdef HAVE_STRING_H
#include <string.h>
#endif

/* static declarations of local routines */
static short inf_by_rand(const mu_ent *const first, const mu_ent *const second);
static short inf_by_track(const mu_ent *const first, const mu_ent *const second);
static short inf_by_posn(const mu_ent *const first, const mu_ent *const second);
static short inf_by_date(const mu_ent *const first, const mu_ent *const second);
static short inf_by_length(const mu_ent *const first, const mu_ent *const second);
static short inf_by_bitrate(const mu_ent *const first, const mu_ent *const second);
static short inf_by_freq(const mu_ent *const first, const mu_ent *const second);
static short inf_by_size(const mu_ent *const first, const mu_ent *const second);
static short inf_by_mtime(const mu_ent *const first, const mu_ent *const second);
static short inf_by_artist(const mu_ent *const first, const mu_ent *const second);
static short inf_by_album(const mu_ent *const first, const mu_ent *const second);
static short inf_by_title(const mu_ent *const first, const mu_ent *const second);
static short inf_by_filetype(const mu_ent *const first, const mu_ent *const second);
static short inf_by_file(const mu_ent *const first, const mu_ent *const second);
static short inf_by_uri(const mu_ent *const first, const mu_ent *const second);
static short inf_by_genre(const mu_ent *const first, const mu_ent *const second);
static short inf_by_dir(const mu_ent *const first, const mu_ent *const second);
static short inf_global(const mu_ent *const first, const mu_ent *const second, const mu_config *const conf);

/**
 * Sorts the list of musical entries.
 *
 * This is a fully home-made quicksort implementation. It places directory
 * entries first, then songs accordingly to the configuration.
 *
 * @param base The first entry of the portion to sort
 * @param end The last entry of the portion to sort
 * @param conf The configuration structure
 *
 * @return The first entry of the sorted list
 */
mu_ent *quicksort(mu_ent *base, mu_ent *end, const mu_config *const conf)
{
	mu_ent *prem;
	mu_ent *deux = base;
	/* The first entry will be used as reference */
	mu_ent *refe = base;

	/* If we have 0 or one entry, sorting is very easy */
	if ((base == end) || (base->next == end))
		return base;

	/* We will begin comparisons with the second entry of the list */
	prem = base->next;

	do {
		/* If the current entry is not correctly placed,
		move it before the reference */
		if (inf_global(prem, refe, conf) < 0) {
			deux->next = prem->next;
			prem->next = base;
			base = prem;
			prem = deux->next;
		/* Otherwise, just continue working with the next one */
		} else {
			deux = prem;
			prem = prem->next;
		}
	} while (prem != end);

	/* We now have two lists, A & B, separated by a value V,
	considering A < V < B . Just run the same quicksort on A and B */
	base = quicksort(base, refe, conf);
	refe->next = quicksort(refe->next, end, conf);

	return base;
}

/**
 * Sets sort order function pointers.
 *
 * @param conf MusicIndex configuration paramaters struct.
 */
void set_fctptrs(mu_config *const conf)
{
	register unsigned short i;

	for (i = 0; i < ARG_NUMBER; i++) {
		switch (conf->order[i]) {
			case SB_ARTIST:
				conf->order_functions[i] = inf_by_artist;
				break;
			case SB_ALBUM:
				conf->order_functions[i] = inf_by_album;
				break;
			case SB_TITLE:
				conf->order_functions[i] = inf_by_title;
				break;
			case SB_TRACK:
				conf->order_functions[i] = inf_by_track;
				break;
			case SB_POSN:
				conf->order_functions[i] = inf_by_posn;
				break;
			case SB_LENGTH:
				conf->order_functions[i] = inf_by_length;
				break;
			case SB_BITRATE:
				conf->order_functions[i] = inf_by_bitrate;
				break;
			case SB_FILETYPE:
				conf->order_functions[i] = inf_by_filetype;
				break;
			case SB_FILENAME:
				conf->order_functions[i] = inf_by_file;
				break;
			case SB_URI:
				conf->order_functions[i] = inf_by_uri;
				break;
			case SB_DATE:
				conf->order_functions[i] = inf_by_date;
				break;
			case SB_SIZE:
				conf->order_functions[i] = inf_by_size;
				break;
			case SB_GENRE:
				conf->order_functions[i] = inf_by_genre;
				break;
			case SB_DIR:
				conf->order_functions[i] = inf_by_dir;
				break;
			case SB_RANDOM:
				conf->order_functions[i] = inf_by_rand;
				break;
			case SB_MTIME:
				conf->order_functions[i] = inf_by_mtime;
				break;
			case SB_FREQ:
				conf->order_functions[i] = inf_by_freq;
				break;
			default:
				conf->order_functions[i] = inf_by_uri;
		}
	}
	if (i == ARG_NUMBER)
		i--;
	/* Force inf_by_uri as last comparison */
	conf->order_functions[i] = inf_by_uri;
}


/* All the following function should return an INCREASING order,
 * unless specified otherwise. */

static short inf_by_rand(const mu_ent *const first, const mu_ent *const second)
{
	return (rand()-(RAND_MAX/2));	/* if rand returns long, the implicit cast will add entropy ;o) */
}

static short inf_by_track(const mu_ent *const first, const mu_ent *const second)
{
	return (first->track - second->track);
}

static short inf_by_posn(const mu_ent *const first, const mu_ent *const second)
{
	return (first->posn - second->posn);
}

static short inf_by_date(const mu_ent *const first, const mu_ent *const second)
{
	return (first->date - second->date);
}

static short inf_by_length(const mu_ent *const first, const mu_ent *const second)
{
	return (first->length - second->length);
}

static short inf_by_bitrate(const mu_ent *const first, const mu_ent *const second)
{
	/* bitrate is long */
	return ((first->bitrate < second->bitrate) ? -1 : 1);
}

/**
 * Sorts by samplerate.
 * @param first first
 * @param second second
 * @return order
 */
static short inf_by_freq(const mu_ent *const first, const mu_ent *const second)
{
	return (first->freq - second->freq);
}

static short inf_by_size(const mu_ent *const first, const mu_ent *const second)
{
	/* size is long */
	return ((first->size < second->size) ? -1 : 1);
}

/**
 * Sorts by mtime.
 * Sort order is inverted here (decreasing), since it wouldn't
 * make sense to have "oldest" entries first.
 * @param first first
 * @param second second
 * @return order
 */
static short inf_by_mtime(const mu_ent *const first, const mu_ent *const second)
{
	/* mtime is long */
	return ((first->mtime < second->mtime) ? 1 : -1);
}

static short inf_by_artist(const mu_ent *const first, const mu_ent *const second)
{
	if ((!first->artist) || (!second->artist)) {
		if (first->artist)
			return -1;
		else if (second->artist)
			return 1;
		else
			return 0;
	}
	return strcasecmp(first->artist, second->artist);
}

static short inf_by_album(const mu_ent *const first, const mu_ent *const second)
{
	if ((!first->album) || (!second->album)) {
		if (first->album)
			return -1;
		else if (second->album)
			return 1;
		else
			return 0;
	}
	return strcasecmp(first->album, second->album);
}

static short inf_by_title(const mu_ent *const first, const mu_ent *const second)
{
	if ((!first->title) || (!second->title)) {
		if (first->title)
			return -1;
		else if (second->title)
			return 1;
		else
			return 0;
	}
	return strcasecmp(first->title, second->title);
}

static short inf_by_filetype(const mu_ent *const first, const mu_ent *const second)
{
	return (first->filetype - second->filetype);
}

static short inf_by_file(const mu_ent *const first, const mu_ent *const second)
{
	return strcasecmp(first->file, second->file);
}

static short inf_by_uri(const mu_ent *const first, const mu_ent *const second)
{
	return strcmp(first->uri, second->uri);
}

static short inf_by_genre(const mu_ent *const first, const mu_ent *const second)
{
	if ((!first->genre) || (!second->genre)) {
		if (first->genre)
			return -1;
		else if (second->genre)
			return 1;
		else
			return 0;
	}
	return strcasecmp(first->genre, second->genre);
}

/* XXX celle la elle meriterait un ptit commentaire... */
static short inf_by_dir(const mu_ent *const first, const mu_ent *const second)
{
	const char *const one = first->uri;
	const char *const two = second->uri;
	unsigned char cone = 'a', ctwo = 'a';
	register unsigned short i;

	for (i=0; one[i] == two[i]; i++)
		continue;

	for (; ((cone!='\0') && (cone!='/')) || ((ctwo!='\0') && (ctwo!='/')); i++) {
		if (((one[i]=='\0')||(one[i]=='/'))&&(cone!='\0')&&(cone!='/'))
			cone = one[i];

		if (((two[i]=='\0')||(two[i]=='/'))&&(ctwo!='\0')&&(ctwo!='/'))
			ctwo = two[i];
	}

	return ((short)cone - (short)ctwo);
}

/**
 * Returns the "difference" between 2 files.
 *
 * If one entry is a dir, put it first. If both are, sort them by alpahbetic
 * order. Then, compare according to each different parameter designed.
 *
 * @param first The first entry
 * @param second The second entry
 * @param conf The configuration structure
 *
 * @return The difference between entries (<0 means order will be inverted)
 */
static short inf_global(const mu_ent *const first, const mu_ent *const second, const mu_config *const conf)
{
	short result;
	register unsigned short i;

	if ((first->filetype < 0) && (second->filetype < 0))
		return (inf_by_file(first, second));
	else if (first->filetype < 0)
		return -1;
	else if (second->filetype < 0)
		return 1;

	for (i = 0; i < ARG_NUMBER; i++) {
		if (conf->order_functions[i]) {
			if ((result = (conf->order_functions[i](first, second))))
				return result;
		}
	}
	return 1;
}
