/*
 *  mod_musicindex.c
 *  mod_musicindex
 *
 *  $Id: mod_musicindex.c 662 2006-07-09 21:05:07Z varenet $
 *
 *  Created by Regis BOUDIN on Sat Dec 28 2002.
 *  Copyright (c) 2002-2006 Regis BOUDIN
 *  Copyright (c) 2002-2005 Thibaut VARENE
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version 2.1,
 *  as published by the Free Software Foundation.
 *
 */


/**
 * @mainpage
 * @section about About
 *	mod_musicindex is an Apache module aimed at being a C implementation
 *	of the Perl module <a href="http://search.cpan.org/dist/Apache-MP3/">
 *	Apache::MP3</a>. It allows nice displaying of directories containing
 *	MP3, Ogg Vorbis, FLAC or MP4 files, including sorting them on various
 *	fields, streaming/downloading them, constructing playlist, and searching.
 *
 * @section features Features
 *	- Allow/deny file downloading.
 *	- Allow/deny file streaming.
 *	- Allow/deny file searching (recursively or not).
 *	- Default sorting order in Apache configuration and dynamic sorting while browsing.
 *	- Random directory on the fly relocation (EXPERIMENTAL).
 *	- Highly configurable data display.
 *	- Custom folder picture (using (.){cover,folder}.{png,jpg,gif} if found).
 *	- Cache subsystem (using flat files).
 *	- Can redirect to an icecast server running in "staticdir" mode.
 *	- Custom cross-directories playlists.
 *	- Multiple CSS support.
 *	- Per directory configuration (using .htaccess).
 *	- XHTML output.
 *	- Archive download (with libarchive - EXPERIMENTAL).
 *	- RSS and Podcasts feeds generation.
 *
 * @section dependencies Dependencies
 * 	- <a href="http://httpd.apache.org/download.cgi">Apache</a> (1.3 or 2).
 *	- <a href="http://downloads.xiph.org/releases/ogg/">libogg</a>.
 *	- <a href="http://downloads.xiph.org/releases/vorbis/">libvorbis</a>.
 *	- <a href="ftp://ftp.mars.org/pub/mpeg/">libmad0</a>.
 *	- <a href="ftp://ftp.mars.org/pub/mpeg/">libid3tag0</a>.
 *	- <a href="http://flac.sourceforge.net/">libflac</a>.
 *	- <a href="http://mpeg4ip.sourceforge.net/">libmp4v2</a>.
 *	- <a href="http://people.freebsd.org/~kientzle/libarchive/">libarchive</a>.
 *
 * @subpage devnotes
 */

/**
 * @page devnotes Developers Notes
 *	Developers want to keep in mind we are aiming at the cleanest code possible.
 *	No matter how wicked what the code does is, said code should do dirty
 *	things with grace and proper documentation ;)
 *
 *	The coding style used throughout the code tries to stick as much as
 *	possible to the GNU Coding Style and/or the Linux Kernel Coding Style.
 *
 *	Developers should also take special care to typing. Strict typing is the
 *	first protection against coding mistakes that might lead to dreadful
 *	crashes. The same applies to function modifier (in particular the
 *	<code>static</code> modifier, that should be used whenever appropriate.
 *
 *	Finally, we try to put Apache's specific arguments to functions at the
 *	beginning of their param list.
 *
 * @section reminders Coding style reminders
 * @subsection pointers Rules about pointers
 *	These are often forgotten, so let's recap how to enforce typing on pointers:
 *	- <code>const char *answer_ptr = "Forty-Two";</code>
 *		- <code>answer_ptr = "Fifty-One";</code>	is legal (answer_ptr is a variable)
 *		- <code>*answer_ptr = 'X';</code>	is illegal (*answer_ptr is a constant)
 *	- <code>char *const name_ptr = "Test";</code>
 *		- <code>name_ptr = "New";</code>	is illegal (name_ptr is constant)
 *		- <code>*name_ptr = 'B';</code>		is legal (*name_ptr is a char)
 *	- <code>const char *const title_ptr = "Title";</code>
 *		- <code>title_ptr = "New";</code>	is illegal (title_ptr is constant)
 *		- <code>*title_ptr = 'X';</code>	is illegal (*title_ptr is a constant)
 *	
 *	Enforcing these simple rules helps code readability (as well as improving
 *	build time checks): in fact, defining a pointer like <code>const char *p;</code>
 *	tells us that this pointer will only be used to navigate in a string, but
 *	won't be used to modify it.
 *
 * @subsection variables Rules about variables
 *	- As a rule of thumb, always try to declare all function variables at the
 *	beginning of the function, and avoid as much as possible to declare
 *	subvariables in blocks.
 *	- In any case, <b>never</b> declare new variables <b>after</b> code in a block.
 *	- Finally, special care should be bore to storage space: always declare
 *	as <i>little</i> as possible: we want the module to consume as few
 *	memory as possible. That also applies to functions return types.
 */
 
/**
 * @file
 * Core file.
 *
 * This file is the core of the module. It contains Apache's mandatory stuff.
 *
 * @author Regis Boudin
 * @version $Revision: 662 $
 * @date 2002-2005
 *
 * http://httpd.apache.org/docs-2.0/developer/API.html
 * http://httpd.apache.org/dev/apidoc/
 *
 * @warning Use it at your own risks!
 *
 * @todo Complete code documentation.
 * @todo (low pri whishlist) On the fly tag rewriting or metadata support.
 * @todo Considering the previous item, we might also need a handler for
 * streaming, otherwise we won't be able to restrain file downloading
 * (Apache::MP3 doesn't do anything better there fwiw).
 * @todo prepare the possibility to generate ices/mpd playlists.
 * @todo i18n (in progress).
 * @todo enforce strict type policy.
 * @todo get rid of str*cmp() whenever possible.
 * @todo review the randomdir implementation.
 *
 * @todo redesign cache integration to speed it up:
 *	- cache init on module initialization
 *	- cache access init (eg: db connex) on conn request
 *	- cache access cleanup (eg: db close) on end conn request
 *	- full real files bypass when possible (eg dir listing). ready
 *	- full cache-dependent search system
 *	(this is mostly done. Need to work on SQL implementation)
 *
 * @todo Check how much overhead a DAAP implementation would bring. The
 *	protocol works on top of HTTP (where we are) and could make use of the
 *	cache.
 *
 * @todo Audit the code: in many places we are using a lot of arguments passed
 * 	to functions that could easily be gotten otherwise by dereferencing a 
 * 	few pointers. Try to figure out what's the most efficient (probably
 * 	passing a lot of args), balancing with storage space usage.
 *
 * @todo Decide how long we want to support Apache 1.3. APR provides lots of
 *	useful features, and the API compatibility between 1.3 and 2.2 is not
 *	certain.
 */

#include "mod_musicindex.h"
#include "playlist.h"
#include "config.h"
#include "html.h"
#include "http.h"
#include "sort.h"

#include <http_core.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#define MUSIC_HEADER_STRING "mod_musicindex/" MUSIC_VERSION_STRING

#ifdef BUILD_FOR_APACHE2
/** mime types handled */
static const char *handlers[] = {
	"audio/mpeg",
	"application/ogg",
	"audio/flac",
	"audio/x-ogg", /* At some point, we should be able to remove this one */
	NULL
};
#else
#include <http_main.h>
#endif

#ifdef ENABLE_OUTPUT_ARCHIVE
extern void send_tarball(request_rec *r, const mu_ent *const p, const mu_config *const conf);
#endif

static int handle_musicindex(request_rec *r)
{
	/* Get the module configuration */
	mu_config *const conf = (mu_config *)ap_get_module_config(r->per_dir_config, &musicindex_module);
	mu_ent *main_list = NULL, *custom_list = NULL;

	/* if the the module is not active, or if request is not a GET or a POST,
	   let the dir to another module */
	if (!(conf->options & MI_ACTIVE))
		return DECLINED;
	if ((r->method_number != M_GET) && (r->method_number != M_POST))
		return DECLINED;

#ifdef BUILD_FOR_APACHE2
	if(strcmp(r->handler, DIR_MAGIC_TYPE))
		return DECLINED;
#endif

	/* before doing anything (checking args, calling subfunctions), check we
	   can open the directory */
	if (access(r->filename, R_OK|X_OK) != 0) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, r,
			MI_LOG_PREFIX "(%s) Can't open directory: %s",
			__func__, r->filename);
		return HTTP_FORBIDDEN;
	}

	r->allowed |= ((1 << M_GET) | (1 << M_POST));

	/* This part mostly comes from mod_autoindex. If the requested dir does
	   not end with a '/', generate a new request with it */
	if (r->uri[0] == '\0' || r->uri[strlen(r->uri) - 1] != '/') {
		char *file;
		if (r->args != NULL)
			file = ap_pstrcat(r->pool, ap_escape_uri(r->pool, r->uri),
						"/", "?", r->args, NULL);
		else
			file = ap_pstrcat(r->pool, ap_escape_uri(r->pool, r->uri),
						"/", NULL);

		ap_table_setn(r->headers_out, "Location", ap_construct_url(r->pool, file, r));
		return HTTP_MOVED_PERMANENTLY;
	}

	/* Deal with optional arguments */
	switch (r->method_number) {
		case M_GET:
			treat_get_args(r, conf);
			break;
		case M_POST:
			treat_post_args(r, conf);
			break;
		default:
			return DECLINED;
	}
	/* Now, the conf structure contains all the flags correctly set and
	   conf->custom_list points to the arguments of the request, either given
	   by get or post method */

	/* XXX this is ugly */
	if (conf->options & MI_RANDOMDIR) {
		conf->options &= ~MI_RANDOMDIR;
		send_randomdir(r, conf);
		return HTTP_MOVED_TEMPORARILY;
	}

	/* build the string containing the cookie content. We need it for
	   webpages and requests to stream the cookie content */
	if (((conf->options & (MI_STREAM|MI_TARBALL)) == 0) || ((conf->options & MI_STREAMRQ) == MI_STREAMCOOKIE) || ((conf->options & MI_DWNLDRQ) == MI_DWNLDCOOKIE))
		cookie_and_stream_work(r, conf);

	if (((conf->options & MI_STREAMRQ) == MI_STREAMLST) ||
		((conf->options & MI_STREAMRQ) == MI_COOKIESTREAM) ||
		((conf->options & MI_DWNLDRQ) == MI_DWNLDLST) ||
		((conf->options & MI_DWNLDRQ) == MI_COOKIEDWNLD)) {
		/* We build a custom playlist, don't bother with current directory */
		main_list = build_custom_list(r, conf);
	}
	else {
		/* Build current directory content (recursive if necessary) */
		main_list = make_music_entry(r, r->pool, NULL, conf, NULL, MI_RECURSIVE);
		main_list = quicksort(main_list, NULL, conf);
	}

	/* Build custom link list, from conf->custom_list string */
	if (((conf->options & MI_STREAM) == 0) && (conf->custom_list != NULL))
		custom_list = build_custom_list(r, conf);

	/* Depending on the request type, set the HTTP headers. By default we
	   return a web page. */
	if (conf->options & MI_STREAM) {
		ap_set_content_type(r, "audio/x-mpegurl");
		ap_table_setn(r->headers_out, "Content-Disposition",
			"filename = \"playlist.m3u\"");
	}
	else if (conf->options & MI_TARBALL) {
		ap_set_content_type(r, "application/x-tar");
		ap_table_setn(r->headers_out, "Content-Disposition",
			"filename = \"playlist.tar\"");
	}
	else if (conf->options & MI_RSS) {
		ap_set_content_type(r, "text/xml; charset=\"utf-8\"");
	}
	else {
		ap_set_content_type(r, "text/html; charset=\"utf-8\"");

		/* If we have a cookie string, send it */
		if (conf->custom_list != NULL)
			ap_table_setn(r->headers_out, "Set-Cookie", conf->custom_list);
	}

	ap_send_http_header(r);

	if (r->header_only)
		return 0;

	if (conf->options & MI_STREAM)
		send_playlist(r, main_list, conf);
	else if (conf->options & MI_RSS)
		send_rss(r, main_list, conf);
#ifdef ENABLE_OUTPUT_ARCHIVE
	else if (conf->options & MI_TARBALL)
		send_tarball(r, main_list, conf);
#endif
	else {
		send_head(r, conf);
		if (conf->search == NULL)
			send_directories(r, main_list, conf);
		send_tracks(r, main_list, conf);
		send_customlist(r, custom_list, conf);
		send_foot(r, conf);
	}

	return 0;
}

/**
 * Handler for requests on music files.
 *
 * At the moment it can only forbid acces to files if neither streaming nor download is allowed.
 *
 * @param r Apache request_rec to get and send data
 *
 * @return Standard HTTP code (see httpd.h for more infos)
 */
static int handle_musicfile(request_rec *r)
{
	mu_config *const conf = (mu_config *)ap_get_module_config(r->per_dir_config, &musicindex_module);

#ifdef BUILD_FOR_APACHE2
	register unsigned short i;
#endif

	if ((r->method_number != M_GET) || (!(conf->options & MI_ACTIVE)))
		return DECLINED;

#ifdef BUILD_FOR_APACHE2
	for (i=0; handlers[i] && strcmp(r->handler, handlers[i]); i++) {
		if (!strcmp(r->handler, handlers[i]))
			break;
	}

	if (handlers[i] == NULL)
		return DECLINED;
#endif

	/* This deals with download requests, that is when there is a request
	   with no get argument. We return DECLINED when :
		- Download is allowed
		- Stream is allowed and no icecast server is set
	   Otherwise, we consider it is a forbidden request. Keep in mind that
	   DECLINED just means the module won't handle the request, whereas
	   HTTP_FORBIDDEN means the request is not allowed, not matter what. */
	/* XXX We should handle fds by ourselves, and set the mime type according
	   to our standards, maybe? */
	if (r->args == NULL) {
		if (conf->options & MI_ALLOWDWNLD)
			return DECLINED;

		if((conf->options & MI_ALLOWSTREAM) && (conf->iceserver == NULL)) {
#if 0 /* XXX RFC 2424 */

#endif
			return DECLINED;
		}

		return HTTP_FORBIDDEN;
	}


	if ((conf->options & MI_ALLOWSTREAM) && (strcmp(r->args, "stream") == 0)) {
		mu_ent	*head = NULL;

		ap_set_content_type(r, "audio/x-mpegurl");
		ap_table_setn(r->headers_out, "Content-Disposition",
			"filename = \"playlist.m3u\"");

		ap_send_http_header(r);

		if (r->header_only)
			return 0;

		head = make_music_entry(r, r->pool, head, conf, NULL, 0);
		send_playlist(r, head, conf);

		return 0;
	}

	return HTTP_FORBIDDEN;
}

#ifndef BUILD_FOR_APACHE2 /* we build for apache 1.3 */

/**
 * Adds the mod signature into Apache's headers.
 *
 * @param s server_rec
 * @param p pool
 */
static void musicindex_init(server_rec *s, apr_pool_t *p)
{
#ifdef HAVE_GETTEXT
	setlocale(LC_ALL, "");
	textdomain("mod_musicindex");
#endif

	ap_add_version_component(MUSIC_HEADER_STRING);
}

/* XXX Shit, that sucks hell, that's another place to edit when adding/removing a handler DAMN! */
static const handler_rec musicindex_handlers[] = {
	{DIR_MAGIC_TYPE, handle_musicindex},
	{"audio/mpeg", handle_musicfile},
	{"audio/x-ogg", handle_musicfile}, /* Backward compatibility */
	{"application/ogg", handle_musicfile},
	{"audio/flac", handle_musicfile},
	{"audio/mp4", handle_musicfile},	/* XXX cette struct sert a quoi exactement? MP4 marchait meme sans ca */
	{NULL}
};

/* XXX init happens after configuration, it should be possible to hook cache init there */
module MODULE_VAR_EXPORT musicindex_module = {
	STANDARD_MODULE_STUFF,
	musicindex_init,		/**< initializer */
	create_musicindex_config,	/**< dir config creater */
	merge_musicindex_configs,	/**< dir merger */
	NULL,				/**< server config */
	NULL,				/**< merge server config */
	musicindex_cmds,		/**< command table */
	musicindex_handlers,		/**< handlers */
	NULL,				/**< filename translation */
	NULL,				/**< check_user_id */
	NULL,				/**< check auth */
	NULL,				/**< check access */
	NULL,				/**< type_checker */
	NULL,				/**< fixups */
	NULL,				/**< logger */
	NULL,				/**< header parser */
	NULL,				/**< child_init */
	NULL,				/**< child_exit */
	NULL				/**< post read-request */
};

#else /* we build for apache 2 */

/**
 * Adds the mod signature into Apache's headers.
 *
 * @param p pool
 * @param plog pool
 * @param ptemp pool
 * @param s server_rec
 */
static int musicindex_init(apr_pool_t *p, apr_pool_t *plog, apr_pool_t *ptemp, server_rec *s)
{
#ifdef HAVE_GETTEXT
	setlocale(LC_ALL, "");
	textdomain("mod_musicindex");
#endif

	ap_add_version_component(p, MUSIC_HEADER_STRING);

	return OK;
}

static void register_hooks(apr_pool_t *p)
{
	/* for directories, set the mod_autoindex to be called after us */
	/* This should not be necessary with V2.0 */
	static const char * const after[] = { "mod_autoindex.c", NULL };
	ap_hook_handler(handle_musicindex, NULL, after, APR_HOOK_MIDDLE);
	ap_hook_handler(handle_musicfile, NULL, NULL, APR_HOOK_MIDDLE);
	ap_hook_post_config(musicindex_init, NULL, NULL, APR_HOOK_LAST);
}

module AP_MODULE_DECLARE_DATA musicindex_module = {
    STANDARD20_MODULE_STUFF,
    create_musicindex_config,	/* create per-directory config structure */
    merge_musicindex_configs,	/* merge per-directory config structures */
    NULL,			/* create per-server config structure */
    NULL,			/* merge per-server config structures */
    musicindex_cmds,		/* command apr_table_t */
    register_hooks		/* register hooks */
};

#endif /* BUILD_FOR_APACHE2 */
