/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "cmdline_hint.h"

#include "chat.h"
#include "custom_input.h"
#include "usergroup.h"
#include "chat_manager.h"
#include "powerkadu.h"
#include "cmdline.h"

CmdLineHint::CmdLineHint(Chat* parent, QStringList matches)
	: QListBox(0, 0, Qt::WStyle_Customize|Qt::WDestructiveClose|Qt::WStyle_NoBorder)
{
	init(parent, matches);
}

CmdLineHint::CmdLineHint(Chat* parent, QStringList matches, QMap<QString,QString> descriptions)
	: QListBox(0, 0, Qt::WStyle_Customize|Qt::WDestructiveClose|Qt::WStyle_NoBorder)
{
	desc = descriptions;
	init(parent, matches);
}

CmdLineHint::~CmdLineHint()
{
	if (hintDesc)
		delete hintDesc;

	if (chat)
	{
		chat->edit()->setFocus();
	}
}

void CmdLineHint::init(Chat* parent, QStringList matches)
{
	// Initial values
	initialized = false;
	chat = parent;
	insertStringList(matches);
	hintDesc = 0;
	connect(chat_manager, SIGNAL(chatDestroying(const UserGroup*)), this, SLOT(handleDestroyingChat(const UserGroup*)));

	// Edits font metrics
	QFontMetrics fm(chat->edit()->currentFont());
	int para, index;
	chat->edit()->getCursorPosition(&para, &index);
	int fWidth = fm.width( chat->edit()->text(para).left(index-1) );

	// Geometry
	resize(160, 100);
	QPoint p;
	p.setX( fWidth+10 );
	p.setY( -96 + fm.height()*para );
	p = chat->edit()->mapToGlobal( p );
	if (p.x()+165 >= QApplication::desktop()->width())
		p.setX( QApplication::desktop()->width() - 165 );

	move( p );
	if (desc.keys().count())
	{
		p.setY( p.y()-24 );
		p.setX( p.x() );
		p = chat->mapFromGlobal( p );
		hintDesc = new CmdDescHint(parent);
		hintDesc->move( p );
		hintDesc->show();
	}

	// Visibility, focus
	show();
	setCurrItem(0);
	setFocus();
	initialized = true;
}

void CmdLineHint::focusOutEvent(QFocusEvent *e)
{
	if (initialized)
		close();
}

void CmdLineHint::keyPressEvent(QKeyEvent *e)
{
	switch (e->key())
	{
		case Qt::Key_Up:
		{
			int cItem = currentItem();
			setCurrItem(--cItem);

			break;
		}
		case Qt::Key_PageUp:
		{
			int cItem = currentItem();
			for ( int i = 0; i < 5; i++ )
				setCurrItem(--cItem);

			break;
		}
		case Qt::Key_Down:
		{
			int cItem = currentItem();
			setCurrItem(++cItem);

			break;
		}
		case Qt::Key_PageDown:
		{
			int cItem = currentItem();
			for ( int i = 0; i < 5; i++ )
				setCurrItem(++cItem);

			break;
		}
		case Qt::Key_Enter:
		case Qt::Key_Return:
		{
			powerKadu->cmds()->put( selectedItem()->text(), chat );
			close();
			break;
		}
		case Qt::Key_Tab:
			break;
		default:
		{
			close();
			break;
		}
	}
}

void CmdLineHint::keyReleaseEvent(QKeyEvent *e)
{
	if (e->key() == Qt::Key_Tab)
		return;
}

void CmdLineHint::handleDestroyingChat(const UserGroup *group)
{
	Chat *c = chat_manager->findChat(group);
	if (c == chat)
	{
		chat = 0;
		close();
	}
}

void CmdLineHint::setCurrItem(int index)
{
	setCurrentItem(index);
	if (hintDesc)
	{
		if ( desc.keys().contains( currentText() ) )
			hintDesc->setText( desc[currentText()] );
		else
			hintDesc->setText( "" );
	}
}

CmdDescHint::CmdDescHint(QWidget *parent)
	: QLineEdit(parent)
{
	setReadOnly(true);
	resize(160, 24);
}
