/*************************************************************************
 * 
 * irmp3 - Multimedia Audio Jukebox for Linux
 * http://irmp3.sourceforge.net
 *
 * $Source: /cvsroot/irmp3/irmp3/src/irmp3d/mod_timidity.c,v $
 * $Id: mod_timidity.c,v 1.11 2004/02/23 20:32:28 boucman Exp $
 *
 * Copyright (C) by Jeremy Rosen <jeremy.rosen@enst-bretagne.fr>
 *
 * Please contact the current maintainer, Jeremy Rosen <jeremy.rosen@enst-bretagne.fr>
 * for information and support regarding irmp3.
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include "config.h"
#include "irmp3tools.h"
#include "irmp3log.h"
#include "irmp3config.h"
#include "irmp3mod.h"
#include "mod_timidity.h"

/*************************************************************************
 * GLOBALS
 */
int mod_timidity_pid    = 0;	// PID of timidity
int mod_timidity_fdsend = 0;	// fd to send commands to timidity
int mod_timidity_fdrecv = 0;	// fd to get responses from timidity
int mod_timidity_fderror = 0;	// fd to get responses from timidity
char mod_timidity_file[512];
#define TIMIDITY_STATUS_HALT	0
#define TIMIDITY_STATUS_STOP	1
#define TIMIDITY_STATUS_PLAY	2
#define TIMIDITY_STATUS_PAUSE	3
int  mod_timidity_status = TIMIDITY_STATUS_HALT;
fd_set mod_timidity_fdset;
int mod_timidity_song_length = 0;
int mod_timidity_song_pos = 0;

/*************************************************************************
 * MODULE INFO
 */
mod_t mod_timidity = {
	"mod_timidity",
	mod_timidity_deinit,	// deinit
	NULL,			// reload
	&mod_timidity_fdset,	// watchfd_set
	NULL,			// poll
	NULL,			// update
	mod_timidity_message,	// message
	mod_timidity_chld,	// SIGCHLD handler
	mod_timidity_init,
	NULL,			// avoid warning
};
/*************************************************************************
 * CATCH SIGNAL OF EXITING PLAYER
 */
void mod_timidity_chld (pid_t pid, int status)
{
	if (pid == mod_timidity_pid) {
		log_printf(LOG_DEBUG, "mod_timidity_chld(): player %d  exited with code %d\n", pid,WEXITSTATUS(status));
		mod_timidity_pid = 0;
		close(mod_timidity_fdrecv);
		mod_timidity_fdrecv=0;
		close(mod_timidity_fdsend);
		mod_timidity_fdsend=0;
		close(mod_timidity_fderror);
		mod_timidity_fderror=0;
	}
}


/*************************************************************************
 * ANSWER A HALT REQUEST
 *
 */
void mod_timidity_halt (void)
{
	if (mod_timidity_fdsend)
		close(mod_timidity_fdsend);
	mod_timidity_fdsend = 0;
	if (mod_timidity_fdrecv)
		close(mod_timidity_fdrecv);
	mod_timidity_fdrecv = 0;
	if (mod_timidity_fderror)
		close(mod_timidity_fderror);
	mod_timidity_fderror = 0;
	if (mod_timidity_pid) {
		log_printf(LOG_DEBUG, "mod_timidity_halt(): killed player pid %d\n", mod_timidity_pid);
		kill(mod_timidity_pid, SIGTERM);
		mod_timidity_pid = 0;
	}
	FD_ZERO(&mod_timidity_fdset);
	mod_timidity.poll = NULL;
        bzero(mod_timidity_file,sizeof(mod_timidity_file));
	mod_timidity_status = TIMIDITY_STATUS_HALT;
	mod_timidity_song_length = 0;
	mod_timidity_song_pos = 0;
}



/*************************************************************************
 * START PLAYER PROCESS
 */
int mod_timidity_launch (void)
{

	// check if player is already running
	if (mod_timidity_status != TIMIDITY_STATUS_HALT) {
		log_printf(LOG_DEBUG, "mod_timidity_start(): mod_timidity_pid is %d; timidity seems to be running!\n", mod_timidity_pid);
		return 0;
	}
	mod_timidity_pid = system_noblock(&mod_timidity_fdsend,&mod_timidity_fdrecv,&mod_timidity_fderror,"%s -ie dummy %s",config_getstr("timidity_binary",TIMIDITY_PATH),config_getstr("timidity_params",""));
	sleep(1);
	if (!mod_timidity_pid || mod_timidity_pid == -1) {
		log_printf(LOG_DEBUG, "mod_timidity_start(): player process didn't start!\n");
		mod_timidity_halt();
		return -1;
	}

	FD_SET(mod_timidity_fdrecv,&mod_timidity_fdset);
	FD_SET(mod_timidity_fderror,&mod_timidity_fdset);
	mod_timidity.poll = mod_timidity_poll;
	return 0;
}

/******************************************
 * ANSWER A STOP REQUEST
 *
 */

void mod_timidity_stop() 
{
	switch(mod_timidity_status) {
		case TIMIDITY_STATUS_HALT:
		case TIMIDITY_STATUS_STOP:
			return;
		case TIMIDITY_STATUS_PLAY:
			sendtext(mod_timidity_fdsend, "r\n");
		case TIMIDITY_STATUS_PAUSE:
			sendtext(mod_timidity_fdsend, " \n");
			mod_timidity_status = TIMIDITY_STATUS_STOP;
			log_printf(LOG_NORMAL, "Playing stopped\n");
			mod_timidity_file[0]=0;
			mod_timidity_song_length = 0;
			mod_timidity_song_pos = 0;
			log_printf(LOG_NOISYDEBUG,"mod_timidity : entered stop state\n");
			return;
		default:
			log_printf(LOG_ERROR,"mod_timidity: unknown status %d\n",mod_timidity_status);
			return;
	}
}
		

/******************************************
 * ANSWER A PLAY REQUEST
 *
 */
	
void mod_timidity_play(char* song) {
	char *tmp;
	switch(mod_timidity_status) {
		case TIMIDITY_STATUS_HALT:
			if (mod_timidity_launch()) {
				mod_sendmsg(MSGTYPE_PLAYER, "error");
				return;
			}
			tmp = malloc(strlen(song) + 7);
			sprintf(tmp,"L\nPLAY %s\n",song);
			sendtext(mod_timidity_fdsend, tmp);
			log_printf(LOG_NORMAL, "Playing '%s'\n", song);
			strncpy(mod_timidity_file, song, sizeof(mod_timidity_file)-1);
			mod_timidity_file[sizeof(mod_timidity_file)-1]=0;
			free(tmp);
			mod_timidity_status= TIMIDITY_STATUS_PLAY;
			log_printf(LOG_NOISYDEBUG,"mod_timidity : entered play state\n");
			return;

		case TIMIDITY_STATUS_PAUSE:
			sendtext(mod_timidity_fdsend, " \n");
		case TIMIDITY_STATUS_STOP:
		case TIMIDITY_STATUS_PLAY:
			tmp = malloc(strlen(song) + 7);
			sprintf(tmp,"L\nPLAY %s\n",song);
			sendtext(mod_timidity_fdsend, tmp);
			log_printf(LOG_NORMAL, "Playing '%s'\n", song);
			strncpy(mod_timidity_file, song, sizeof(mod_timidity_file)-1);
			mod_timidity_file[sizeof(mod_timidity_file)-1]=0;
			free(tmp);
			mod_timidity_status= TIMIDITY_STATUS_PLAY;
			log_printf(LOG_NOISYDEBUG,"mod_timidity : entered play state\n");
			return;
			
		default:
			log_printf(LOG_ERROR,"mod_timidity: unknown status %d\n",mod_timidity_status);
			return;
	}
}

/******************************************
 * ANSWER A PAUSE REQUEST
 *
 */
	
void mod_timidity_pause() {
	switch(mod_timidity_status) {
		case TIMIDITY_STATUS_HALT:
		case TIMIDITY_STATUS_STOP:
			return;
			
		case TIMIDITY_STATUS_PAUSE:
			sendtext(mod_timidity_fdsend, " \n");
			log_printf(LOG_NOISYDEBUG,"mod_timidity : entered play state\n");
			mod_timidity_status= TIMIDITY_STATUS_PLAY;
			return;
			
		case TIMIDITY_STATUS_PLAY:
			sendtext(mod_timidity_fdsend, " \n");
			log_printf(LOG_NOISYDEBUG,"mod_timidity : entered pause state\n");
			mod_timidity_status= TIMIDITY_STATUS_PAUSE;
			return;
			
		default:
			log_printf(LOG_ERROR,"mod_timidity: unknown status %d\n",mod_timidity_status);
			return;

	}
}
/******************************************
 * ANSWER A SEEK REQUEST
 *
 */
	
void mod_timidity_seek(char* position) {
	switch(mod_timidity_status) {
		case TIMIDITY_STATUS_HALT:
		case TIMIDITY_STATUS_STOP:
			return;

		case TIMIDITY_STATUS_PAUSE:
			sendtext(mod_timidity_fdsend, " \n");
			log_printf(LOG_NOISYDEBUG,"mod_timidity : entered play state\n");
			mod_timidity_status= TIMIDITY_STATUS_PLAY;
		case TIMIDITY_STATUS_PLAY:
			if (*position=='-' || *position=='+')
				sendtext(mod_timidity_fdsend, "f %s\n", position);
			else {
				int time = mod_timidity_song_pos + atoi(position);
				time = time > 0?time:0;
				// after end of song is reported properly by timidity
				sendtext(mod_timidity_fdsend, "f %d\n", time);
			}
			return;

		default:
			log_printf(LOG_ERROR,"mod_timidity: unknown status %d\n",mod_timidity_status);
			return;
	}
}




/*************************************************************************
 * POLL INPUT DATA
 */
void mod_timidity_poll (int fd)
{
	char s[512],*c1,*c2;
	int rc;

	// read the response from the player
	rc = readline(fd, s, sizeof(s));
	if (rc < 0) {
		// player closed connection
		log_printf(LOG_DEBUG,"mod_timidity_poll(): player closed connection with %s.\n",strerror(errno));
		mod_timidity_halt();
		mod_sendmsg(MSGTYPE_PLAYER, "error");
		return;
	}

	// process player response
	// 
	
	c1 = strtok(s, " \t)");
	if(!c1) return;
	c1 += 10; // remove leading timidity

	if (!strcasecmp(c1, "VERSION")) {
		// player version received
		c2 =  strtok(NULL, ")");
		mod_sendmsgf(MSGTYPE_INFO, "player version %s", c2);

	} else if (!strcasecmp(c1, "NEXT")) {
		log_printf(LOG_NORMAL, "End of song reached\n");
		mod_sendmsg(MSGTYPE_PLAYER, "endofsong");
		mod_timidity_stop();
	} else if (!strcasecmp(c1, "TIME")) {
		c2 =  strtok(NULL, ")");
		mod_timidity_song_length = atoi(c2);
		mod_sendmsgf(MSGTYPE_PLAYER, "playing");
		mod_sendmsgf(MSGTYPE_PLAYER, "time 0 %d", mod_timidity_song_length);
	} else if (!strcasecmp(c1, "CURT")) {
		c2 =  strtok(NULL, ")");
		if(mod_timidity_song_pos == atoi(c2)) return;
		mod_timidity_song_pos = atoi(c2);
		mod_sendmsgf(MSGTYPE_PLAYER, "time %d %d",mod_timidity_song_pos, mod_timidity_song_length - mod_timidity_song_pos);
	} else if (!strcasecmp(c1, "CMSG")) {
		c2 =  strtok(NULL, " \t)");
		if(atoi(c2) == 2) { //error
			mod_sendmsg(MSGTYPE_PLAYER,"error");
			mod_timidity_halt();
		}
	} else {
		// unknown message, probably an error
		c2 =  strtok(NULL, "");
		log_printf(LOG_VERBOSE, "timidity reports: '%s %s'\n", c1, c2);
		if(!strcasecmp("yes",config_getstr("abort_on_unknown_timidity_status","no"))) {
			log_printf(LOG_NORMAL, "Playing stopped\n");
			mod_sendmsg(MSGTYPE_PLAYER, "error");
			mod_timidity_halt();
		}
	}
}



/*************************************************************************
 * RECEIVE MESSAGE
 */
void mod_timidity_message (int msgtype, char *msg,const char __attribute__((unused))*sender)
{
	char *c1, *c2, *c3;

	if (msgtype == MSGTYPE_PLAYER) {
		c1 = msg ? strtok(msg, " \t") : NULL;
		if(!c1) return;

		// see if we can play the requested song
		if ( !strcasecmp(c1, "play")) {
			c2 =  strtok(NULL, " \t");
			if(!c2) return;
			c3 = strtok(NULL, "");
			if(!c3) return;
			if ( !strcasecmp(c2,"mod_timidity") ) {
				mod_timidity_play(c3);
			} else {
					log_printf(LOG_DEBUG, "mod_timidity_message(): not for me: '%s'\n", c2);
					mod_timidity_halt();
			}
		} else if (!strcasecmp(c1, "stop") && mod_timidity_pid) {
			mod_timidity_stop();
		} else if (!strcasecmp(c1, "pause") && mod_timidity_pid) {
			mod_timidity_pause();

		} else if (!strcasecmp(c1, "seek") && mod_timidity_pid) {
			c2 =  strtok(NULL, " \t");
			if(!c2) return;
			log_printf(LOG_DEBUG, "mod_timidity_message(): seeking %s\n", c2);
			mod_timidity_seek(c2);
		}
	} else if (msgtype == MSGTYPE_QUERY) {
		c1 = msg ? strtok(msg, " \t") : NULL;
		if(!c1) return;
		if (!strcasecmp(c1,"whocanplay")) {
			c2 =  strtok(NULL, " \t");
			if(!c2) return;
			if(  !strcasecmp(c2,"midi")) {
				mod_sendmsgf(MSGTYPE_INFO,"canplay mod_timidity %s",c2);
			} else if(  !strcasecmp(c2,"recompose")) {
				mod_sendmsgf(MSGTYPE_INFO,"canplay mod_timidity %s",c2);
			} else if(  !strcasecmp(c2,"mod")) {
				mod_sendmsgf(MSGTYPE_INFO,"canplay mod_timidity %s",c2);
			} else if(  !strcasecmp(c2,"kar")) {
				mod_sendmsgf(MSGTYPE_INFO,"canplay mod_timidity %s",c2);
			} else if(  !strcasecmp(c2,"http")) {
				mod_sendmsgf(MSGTYPE_INFO,"canplay mod_timidity %s",c2);
			}
		} else {
			if (mod_timidity_pid) {
				if( !strcasecmp(c1,"midifile")) mod_sendmsgf(MSGTYPE_INFO,"file %s",mod_timidity_file);
				if( !strcasecmp(c1,"time"))    mod_sendmsgf(MSGTYPE_PLAYER, "time %d %d",mod_timidity_song_pos, mod_timidity_song_length - mod_timidity_song_pos);
			}
		}
	}
}


/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_timidity_init (void)
{
	FD_ZERO(&mod_timidity_fdset);

	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_timidity_deinit (void)
{
	mod_timidity_halt();
	log_printf(LOG_DEBUG, "mod_timidity_deinit(): deinitialized\n");
}


/*************************************************************************
 * EOF
 */
