/*************************************************************************
 * 
 * irmp3 - Multimedia Audio Jukebox for Linux
 * http://irmp3.sourceforge.net
 *
 * $Source: /cvsroot/irmp3/irmp3/src/irmp3d/mod_fifo.c,v $ -- unix domain control socket
 * $Id: mod_fifo.c,v 1.9 2004/02/17 20:40:36 boucman Exp $
 *
 * Copyright (C) by Jeremy Rosen, Volker Wegert
 *
 * Please contact the current maintainer, Jeremy Rosen <jeremy.rosen@enst-bretagne.fr>
 * for information and support regarding irmp3.
 *
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>

#include "config.h"
#include "irmp3config.h"
#include "irmp3tools.h"
#include "irmp3log.h"
#include "irmp3mod.h"
#include "mod_fifo.h"

/*************************************************************************
 * GLOBALS
 */
fd_set		mod_fifo_fdset;
int 		mod_fifo_fd;
char 		filename[50];

/*************************************************************************
 * MODULE INFO
 */
mod_t mod_fifo = {
	"mod_fifo",
	mod_fifo_deinit,	// our deinit function
	NULL,			// called when got SIGHUP
	&mod_fifo_fdset,	// we're watching a fifo
	mod_fifo_poll,		// and process the sent text
	NULL,			// we don't need to be called periodically
	NULL,			// our message handler
	NULL,			// SIGCHLD handler
	mod_fifo_init,
	NULL,			// avoid warning
};


/*************************************************************************
 * POLL INPUT DATA
 */
void mod_fifo_poll (int fd)
{
  char s[512];
  readline(fd, s, sizeof(s));
  log_printf(LOG_NOISYDEBUG, "mod_fifo_poll(): got input: '%s'\n", s);
  if (s && *s)
    mod_sendmsg(MSGTYPE_INPUT, s);
}


/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_fifo_init (void)
{
  char *directory, *confname;
  char pid_str[MOD_FIFO_PID_SIZE], uid_str[MOD_FIFO_UID_SIZE], name[MOD_FIFO_FILENAME_MAX_LENGTH];
  int perms, pos_confname, pos_name; 
  unsigned int pos_tmp;
  
  FD_ZERO(&mod_fifo_fdset);
  
  // check whether mod_fifo is enabled at all
  if (!strcasecmp(config_getstr("fifo_enabled", "no"), "yes")) {

    log_printf(LOG_DEBUG, "mod_fifo_init(): initializing\n");
    
    // determine fifo name
    directory = config_getstr("fifo_directory","/tmp");
    confname  = config_getstr("fifo_name", "irmp3-%u-%p.tmp");
    snprintf(pid_str, MOD_FIFO_PID_SIZE, "%d", getpid());
    snprintf(uid_str, MOD_FIFO_UID_SIZE, "%u", getuid());
    pos_name     = 0;
    pos_confname = 0;
    while (*(confname + pos_confname) != 0) {
      if (*(confname + pos_confname) == '%') {
	// process replacement string
	pos_tmp = 0;
	switch (*(confname + (++pos_confname))) {
	case 'u': while (pos_tmp < strlen(uid_str)) { name[pos_name++] = *(uid_str + pos_tmp++); } break;
	case 'p': while (pos_tmp < strlen(pid_str)) { name[pos_name++] = *(pid_str + pos_tmp++); } break;
	}
	pos_confname++;
      } else {
	// copy content
	name[pos_name++] = confname[pos_confname++];
      }
    }
    name[pos_name] = 0;

    sprintf(filename, "%s/%s", directory, name);
    
    // create new fifo?
    if (!strcasecmp(config_getstr("fifo_create", "yes"), "yes")) {
      perms = config_getnum("fifo_perms", 0622);
      // assure we have read access, everything else is ridiculous
      perms |= S_IRUSR;
      log_printf(LOG_DEBUG, "mod_fifo_init(): attempting to create fifo %s with permissions 0%o\n", filename, perms);
      if (mkfifo(filename, perms) < 0) {
	if (errno == EEXIST) {
	  log_printf(LOG_VERBOSE, "mod_fifo_init(): fifo exists %s\n", filename);
	} else {
	  log_printf(LOG_ERROR, "mod_fifo_init(): Unable to create fifo %s\n", filename);
    	  return "Unable to create fifo.";
	}
      }
    }
    
    // open fifo
    // we need to guarentee a writer because of select
    mod_fifo_fd = open(filename, O_RDWR | O_NONBLOCK);
    if (mod_fifo_fd < 0) {
      log_printf(LOG_ERROR, "mod_fifo_init(): Unable to open fifo %s\n", filename);
      return "Unable to open fifo.";
    }

    log_printf(LOG_DEBUG, "mod_fifo_init(): fifo file descriptor is %d\n", mod_fifo_fd);
    FD_SET(mod_fifo_fd, &mod_fifo_fdset);
    
    
  }
  return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_fifo_deinit (void)
{
  // clean up fifo
  if (mod_fifo_fd > 0) {
    log_printf(LOG_DEBUG, "mod_fifo_deinit(): closing fifo...\n");
    close(mod_fifo_fd);
  }
  
  if (!strcasecmp(config_getstr("fifo_remove", "yes"), "yes")) {
    log_printf(LOG_DEBUG, "mod_fifo_deinit(): removing fifo %s\n", filename);
    unlink(filename);
  }
  
  log_printf(LOG_DEBUG, "mod_fifo_deinit(): deinitialized\n");
}



/*************************************************************************
 * EOF
 */
