/*  Copyright (C) 2007 Bernhard R. Link
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02111-1301  USA
 */
#include <config.h>

#include <sys/types.h>
#include <limits.h>
#include <assert.h>
#include <stdbool.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <poll.h>
#include <dirent.h>
#include <regex.h>
#include <getopt.h>
#include <syslog.h>
#include <signal.h>
#include <sys/wait.h>
#ifdef HAVE_SYS_INOTIFY_H
#include <sys/inotify.h>
#else
#ifdef HAVE_INOTIFYTOOLS_INOTIFY_H
#include <inotifytools/inotify.h>
#endif
#endif

#define isbitset(v,b) ((v&(b))!=0)
#define isanybitset(v,b) ((v&(b))!=0)
#define isallbitset(v,b) ((v&(b))==(b))

static const char *program;
static bool foreground = false;
static bool amdaemon = false;
static FILE *logfile = NULL;
static volatile bool termsignaled = false;
static char *pidfilename = NULL;

static void closeallfds(int notifd, int logfd, int dirfd, bool warn) {
	long maxopen;
	int fd;

	maxopen = sysconf(_SC_OPEN_MAX);
	if( maxopen > INT_MAX )
		maxopen = INT_MAX;
	if( maxopen < 0 )
		maxopen = 1024;
	for( fd = 3 ; fd < (int)maxopen ; fd++ ) {
		if( fd != notifd && fd != logfd && fd != dirfd ) {
			if( close(fd) == 0 && foreground && warn ) {
				fprintf(stderr, "File descriptor %d was still open!\n",
					fd);
			}
		}
	}
	errno = 0;
}

static void termsignal(/*@unused@*/ int i __attribute__((unused))) {
	termsignaled = true;
}
static void dolog(const char *format, ...) __attribute__((format(printf,1,2)));
static void logerror(const char *format, ...) __attribute__((format(printf,1,2)));
static void logerror_die(const char *format, ...) __attribute__((format(printf,1,2))) __attribute__((noreturn));
static void dolog(const char *format, ...) {
	va_list ap;

	va_start(ap, format);
	if( amdaemon ) {
		if( logfile != NULL )
			vfprintf(logfile, format, ap);
		else
			vsyslog(LOG_INFO, format, ap);
	} else {
		vfprintf(stdout, format, ap);
	}
	va_end(ap);
}
static void logerror(const char *format, ...) {
	va_list ap;

	va_start(ap, format);
	if( amdaemon ) {
		if( logfile != NULL )
			vfprintf(logfile, format, ap);
		else
			vsyslog(LOG_ERR, format, ap);
	} else {
		fputs(program, stderr);
		fputc(':', stderr);
		fputc(' ', stderr);
		vfprintf(stderr, format, ap);
	}
	va_end(ap);
}
static void logerror_die(const char *format, ...) {
	va_list ap;

	va_start(ap, format);
	if( amdaemon ) {
		if( logfile != NULL )
			vfprintf(logfile, format, ap);
		else
			vsyslog(LOG_ERR, format, ap);
	} else {
		fputs(program, stderr);
		fputc(':', stderr);
		fputc(' ', stderr);
		vfprintf(stderr, format, ap);
	}
	exit(EXIT_FAILURE);
}

static bool waitforevent(int notifd, const int timeout) {
	struct pollfd pollinfo;
	int e, r;

	pollinfo.fd = notifd;
	pollinfo.events = POLLIN;
	pollinfo.revents = 0;
	do {
		errno = 0;
		r = poll(&pollinfo, 1, timeout);
		e = errno;
	} while( r < 0 && e == EINTR && !termsignaled );
	if( termsignaled )
		return false;
	if( r < 0 ) {
		logerror("Error polling inotify descriptor: %s(%d)\n",
				strerror(e),e);
		exit(EXIT_FAILURE);
	}
	if( r == 0 ) {
		if( (pollinfo.revents & POLLIN) == 0 )
			return false;
	}
	if( (pollinfo.revents & (POLLERR|POLLHUP|POLLNVAL)) != 0 ) {
		logerror("Where did my inotify descriptor go?\n");
		exit(EXIT_FAILURE);
	}
	return (pollinfo.revents & (POLLIN|POLLPRI)) != 0;
}


static struct inotify_event *getevent(int notifd, bool *mayread) {
	static void *buffer = NULL;
	static size_t bufcount = 0;
	static size_t already = 0;
	struct inotify_event *event;
	ssize_t bytesread;
	size_t size;
	bool doread = *mayread;

	if( already >= sizeof(struct inotify_event) ) {
		size = sizeof(struct inotify_event)
		       + ((struct inotify_event*)buffer)->len;
		if( already >= size )
			doread = false;
	}
	if( doread ) {
		if( bufcount <= already+1024 ) {
			buffer = realloc(buffer, already+20480);
			if( buffer == NULL )
				return NULL;
			else
				bufcount = already+20480;
		}
		assert( buffer != NULL );
		assert( bufcount > already);

		bytesread = read(notifd, buffer+already, bufcount-already);
		*mayread = false;
		if( bytesread < 0 ) {
			int e = errno;
			logerror("Error reading from inotify descriptor: %s(%d)\n",
					strerror(e), e);
			exit(EXIT_FAILURE);
		}
		assert( (size_t)bytesread <= bufcount - already );
		already += bytesread;
	}
	if( already < sizeof(struct inotify_event) )
		return NULL;
	size = sizeof(struct inotify_event)+((struct inotify_event*)buffer)->len;
	if( already < size )
		return NULL;
	event = malloc(size);
	if( event == NULL ) {
		logerror("Out of memory! (allocating %lu bytes)\n",
				(unsigned long)size);
		already -= size;
		memmove(buffer, buffer+size, already);
		return NULL;
	}
	memcpy(event, buffer, size);
	already -= size;
	memmove(buffer, buffer+size, already);
	return event;
}

static struct lookfor {
	struct lookfor *next;
	char *regexp;
	char *prefix, *suffix;
	size_t prefixlen, suffixlen;
	char *dir;
	char *action;
	struct argument {
		char *argument;
		size_t len;
		size_t substitute;
	} *arguments;
	size_t num_arguments;
	bool logstderr, logstdout;
	regex_t reg;
} *lookfor = NULL;

static char *fill_str(const char *value, size_t len, size_t subst, const char *filename) {
	size_t fnl = strlen(filename);
	char *r;

	r = malloc(len+fnl+1);
	if( r == NULL )
		return r;
	memcpy(r, value, subst);
	memcpy(r+subst, filename, fnl);
	memcpy(r+subst+fnl, value+subst, 1+len-subst);
	return r;
}

static void doaction(const struct lookfor *l, const char *filename) {
	char **args;
	pid_t child;
	size_t i;

	// TODO: what about getting output from the childs and mailing
	// it somewhere?

	// todo: only if verbose and with full arguments (and filename escaped)...
	dolog("Will call action %s for: %s\n", l->action, filename);

	child = fork();
	if( child == (pid_t)-1 ) {
		int e = errno;
		logerror("Error forking: %s(%d)\n",
				strerror(e), e);
		return;
	}
	if( child > 0 ) {
		int status;
		pid_t w;
		do {
			w = waitpid(child, &status, 0);
		} while( w == (pid_t)-1 && errno==EINTR && !termsignaled );
		if( w == child ) {
			if( WIFEXITED(status) && WEXITSTATUS(status) != 0 ) {
				// todo: full escaped action string...
				logerror("Action '%s' returned exitcode %d\n",
						l->action,
						(int)(WEXITSTATUS(status)));
			} else if( WIFSIGNALED(status) ) {
				// todo: full escaped action string...
				logerror("Action '%s' died by signal %d\n",
						l->action,
						(int)(WTERMSIG(status)));
			}
		}
		return;
	}
	/* in the fork */
	args = calloc((l->num_arguments+2), sizeof(char *));
	if( args == NULL )
		logerror_die("Out of memory in child!\n");
	args[0] = l->action;
	for( i = 0 ; i < l->num_arguments; i++ ) {
		struct argument *a = &l->arguments[i];
		if( a->substitute != (size_t)-1 ) {
			args[i+1] = fill_str(a->argument, a->len,
					a->substitute, filename);
			if( args[i+1] == NULL )
				logerror_die("Out of memory in child!\n");
		} else
			args[i+1] = a->argument;
	}
	args[l->num_arguments+1] = NULL;
	if( l->dir != NULL && chdir(l->dir) != 0 ) {
		int e = errno;
		logerror_die("Could not change to directory '%s' in child: %s(%d)\n",
				l->dir, strerror(e), e);
	}
	if( l->logstdout && logfile != NULL && dup2(fileno(logfile), 1) == -1 ) {
		int e = errno;
		logerror_die("Could not redirect stdout to logfile: %s(%d)\n",
				strerror(e), e);
	}
	if( l->logstderr && logfile != NULL && dup2(fileno(logfile), 2) == -1 ) {
		int e = errno;
		logerror_die("Could not redirect stderr to logfile: %s(%d)\n",
				strerror(e), e);
	}
	if( logfile != NULL)
		fclose(logfile);
	closeallfds(-1, -1, -1, false);
	execvp(args[0],args);
}

static bool shouldprocess(const struct lookfor *l, const char *filename) {
	size_t len = strlen(filename);

	/* does not start or end as it should */
	if( l->prefix != NULL && (len < l->prefixlen ||
			memcmp(filename, l->prefix, l->prefixlen) != 0 ))
		return false;
	if( l->suffix != NULL && (len < l->suffixlen ||
			memcmp(filename+len-l->suffixlen,
			       l->suffix, l->suffixlen) != 0 ))
		return false;
	/* if there is a regexp, and it does not match, don't */
	if( l->regexp != NULL ) {
		if( regexec(&l->reg, filename, 0, NULL, 0) != 0 )
			return false;
	}
	return true;
}

static void processevent(const struct inotify_event *event) {
	const struct lookfor *l;

	/* report errors and possible problems */
	if( isanybitset(event->mask,IN_DELETE_SELF|IN_UNMOUNT) ) {
		logerror_die("Watched directory vanished!\n");
	}
	if( isbitset(event->mask,IN_Q_OVERFLOW) ) {
		logerror("Warning: overflow happened! Events might have been missed.\n");
	}
// debug stuff, to be removed or conditialized:
//	if( isbitset(event->mask,IN_CREATE) ) {
//		dolog("File was created: %s\n", event->name);
//	}
//	if( isbitset(event->mask,IN_CLOSE_WRITE) ) {
//		dolog("File ready: %s\n", event->name);
//	}
//	if( isbitset(event->mask,IN_MOVED_TO) ) {
//		dolog("Full file moved here: %s\n", event->name);
//	}
	/* look if anything intresting is happening */

	if( !isanybitset(event->mask,IN_MOVED_TO|IN_CLOSE_WRITE) )
		return;

	for( l = lookfor ; l != NULL ; l = l->next ) {
		if( shouldprocess(l, event->name) )
			doaction(l, event->name);
	}
}

static void process_directory(DIR *dir) {
	const struct lookfor *l;
	struct dirent *ent;

	while( (ent = readdir(dir)) != NULL ) {
		if( ent->d_type != DT_REG )
			continue;
		for( l = lookfor ; l != NULL ; l = l->next ) {
			if( shouldprocess(l, ent->d_name) )
				doaction(l, ent->d_name);
		}
	}
}

static inline char *xstrdup(const char *s) {
	char *v;
	v = strdup(s);
	if( v == NULL )
		logerror_die("Out of memory!\n");
	return v;
}

static void parse_regexp(struct lookfor *l, const char *regexp) {
	int r;
	if( l->regexp != NULL )
		logerror_die("Multiple regular expressions for the same action!\n");
	l->regexp = xstrdup(regexp);
	r = regcomp(&l->reg, l->regexp, REG_EXTENDED|REG_NOSUB);
	if( r != 0 ) {
		size_t s;
		char *msg;
		s = regerror(r, &l->reg, NULL, 0);
		msg = malloc(s);
		if( msg == NULL )
			logerror_die("Out of memory!\n");
		regerror(r, &l->reg, msg, s);
		logerror_die("Error parsing regular expression '%s': %s\n",
				l->regexp, msg);
	}
}

static void parse_argument(struct argument *a, const char *v) {
	size_t len;
	const char *p;
	char *q;
	bool hadsubst = false, hadescaped = false;

	len = 0; p = v;
	while( *p != '\0' ) {
		if( p[0] == '{' && p[1] == '{' && p[2] == '}' && p[3] == '}' ) {
			hadescaped = true;
			len += 2;
			p += 4;
		} else if( p[0] == '{' && p[1] == '}' ) {
			if( hadsubst )
				logerror_die("Only one {} per argument is supported!\n");
			hadsubst = true;
			p +=2;
		} else {
			len++;
			p++;
		}
	}

	a->len = len;
	a->substitute = (size_t)-1;
	a->argument = malloc(len+1);
	if( a->argument == NULL )
		logerror_die("Out of memory!\n");
	if( !hadsubst && !hadescaped )
		memcpy(a->argument, v, len+1);
	else {
		p = v; q = a->argument;
		while( *p != '\0' ) {
			if( p[0] == '{' && p[1] == '{' && p[2] == '}' && p[3] == '}' ) {
				assert( q < a->argument + len + 1 );
				*(q++) = '{';
				*(q++) = '}';
				p += 4;
			} else if( p[0] == '{' && p[1] == '}' ) {
				p += 2;
				a->substitute = q-a->argument;
			} else {
				assert( q < a->argument + len );
				*(q++) = *(p++);
			}
		}
		assert( q == a->argument + len );
		*q = '\0';
	}
}

static void parseaction(int argc, char * const argv[], bool havelogfile) {
	int i, j, opt;
	static const struct option long_options[] = {
		{ "stdout-to-log", no_argument, NULL, 1},
		{ "stderr-to-log", no_argument, NULL, 2},
		{ "regexp", required_argument, NULL, 'r'},
		{ "prefix", required_argument, NULL, 'p'},
		{ "suffix", required_argument, NULL, 's'},
		{ "chdir", required_argument, NULL, 'c'},
		{ NULL,	0, NULL, 0}
	};
	struct lookfor *l = calloc(1,sizeof(struct lookfor));

	if( l == NULL ) {
		fprintf(stderr, "%s: Out of memory.\n", program);
		exit(EXIT_FAILURE);
	}
	l->next = lookfor;
	lookfor = l;
	while( (opt = getopt_long(argc, argv, "+p:s:r:c:", long_options, NULL)) >= 0 ) {
		switch( opt ) {
			case 1:
				if( !havelogfile )
					logerror_die("Action's --stderr-to-log needs global --logfile option!\n");
				l->logstdout = true;
				break;
			case 2:
				if( !havelogfile )
					logerror_die("Action's --stderr-to-log needs global --logfile option!\n");
				l->logstderr = true;
				break;
			case 'r':
				parse_regexp(lookfor, optarg);
				break;
			case 'p':
				if( lookfor->prefix != NULL )
					logerror_die("Multiple --prefix options!\n");
				lookfor->prefix = xstrdup(optarg);
				lookfor->prefixlen = strlen(lookfor->prefix);
				break;
			case 's':
				if( lookfor->suffix != NULL )
					logerror_die("Multiple --suffix options!\n");
				lookfor->suffix = xstrdup(optarg);
				lookfor->suffixlen = strlen(lookfor->suffix);
				break;
			case 'c':
				if( lookfor->dir != NULL )
					logerror_die("Multiple --chdir options!\n");
				lookfor->dir = xstrdup(optarg);
				break;
			case '?':
				if( optopt == '\0' )
					logerror_die("Unknown/ambiguous action option '%s'!\n",
							argv[optind-1]);
				else if( optopt == 'p' || optopt == 'l' )
					logerror_die("Action option '%s' needs an argument!\n",
							argv[optind-1]);
				else
					logerror_die("Unknown action option '%c'!\n",
							(char)optopt);
				break;
		}
	}
	if( optind >= argc )
		logerror_die("Unexpected end of command line (action expected)!\n");
	l->action = xstrdup(argv[optind]);
	optind++;
	if( l->action == NULL )
		logerror_die("Out of memory!\n");
	i = optind;
	while( i < argc && strcmp(argv[i],";") != 0 )
		i++;
	if( i >= argc )
		logerror_die("Missing ';' at end of action!\n"
			 "(Don't forget to escape it (\\;) if you use a shell)\n");
	l->num_arguments = i-optind;
	l->arguments = calloc(l->num_arguments, sizeof(struct argument));
	if( l->arguments == NULL )
		logerror_die("Out of memory!\n");
	for( j = optind; j < i ; j++ ) {
		parse_argument(&l->arguments[j-optind], argv[j]);
	}
	optind = i+1;
}


static void syntax(FILE *file, int exitcode) {
	fprintf(file,
"Syntax: %s [<options>*] <directory to watch> <actions>\n"
"where option can be:\n"
"\t--logfile <file>\n"
"\t--pid-file <file>\n"
"\t--foreground\n"
"\t--initialsearch\n"
"and each action is [<action-options>*] command [<arguments>] ;\n"
"with possible action-options:\n"
"\t--prefix <string>\tonly processes filenames starting with string.\n"
"\t--suffix <string>\tonly processes filenames ending with string.\n"
"\t--regexp <regular expression>\t... matching the regular expression given.\n"
"\t--chdir <directory>\tchange directory before calling action.\n"
"\t--stdout-to-log\t\tsend output to the logfile\n"
"\t--stderr-to-log\t\tsend error output to the logfile\n"
"In the arguments {} is replaced by the filename found.\n"
"Note the ; at the end of every action. (In a shell, use \\; to get ;)\n"
, program);
	exit(exitcode);
}

int main(int argc, char * const argv[]) {
	int notifd;
	int dirwatchid;
	const char *directory;
	char *logfilename = NULL;
	int timeout, e, opt;
	struct inotify_event *event;
	FILE *pidfile;
	DIR *dir;
	bool initialsearch = false;
	static const struct option global_long_options[] = {
		{ "version",	no_argument,		NULL, 1},
		{ "help",	no_argument,		NULL, 'h'},
		{ "logfile",	required_argument,	NULL, 'l'},
		{ "foreground",	no_argument,		NULL, 'F'},
		{ "initialsearch",no_argument,		NULL, 'i'},
		{ "pid-file",	required_argument,	NULL, 'p'},
		{ NULL,	0, NULL, 0}
	};

	program = argv[0];
	if( argc < 2 ) {
		logerror("Not enough arguments.\n");
		syntax(stderr, EXIT_FAILURE);
	}
	opterr = 0;
	while( (opt = getopt_long(argc, argv, "+h?p:l:Ffi", global_long_options, NULL)) >= 0 ) {
		switch( opt ) {
			case 1:
				puts(PACKAGE " " VERSION);
				exit(EXIT_SUCCESS);
				break;
			case 'h':
				syntax(stdout, EXIT_SUCCESS);
				break;
			case 'l':
				if( logfile != NULL )
					logerror_die("Multiple --logfile options are not supported.\n");
				logfilename = xstrdup(optarg);
				break;
			case 'p':
				if( pidfilename != NULL )
					logerror_die("Multiple --pid-file options are not supported.\n");
				pidfilename = xstrdup(optarg);
				break;
			case 'f':
			case 'F':
				foreground = true;
				break;
			case 'i':
				initialsearch = true;
				break;
			case '?':
				if( optopt == '\0' )
					logerror_die("Unknown/ambiguous global option '%s'!\n",
							argv[optind-1]);
				else if( optopt == 'p' || optopt == 'l' )
					logerror_die("Global option '%s' needs an argument!\n",
							argv[optind-1]);
				else
					logerror_die("Unknown global option '%c'!\n",
							(char)optopt);
				break;
		}
	}
	if( optind >= argc)
		logerror_die("Not enough arguments!\n");
	directory = argv[optind];
	optind++;
	while( optind < argc )
		parseaction(argc, argv, logfilename != NULL);
	if( lookfor == NULL )
		logerror_die("Nothing to do specified!\n");

	notifd = inotify_init();
	if( notifd < 0 ) {
		e = errno;
		fprintf(stderr, "%s: Could not initialize inotify: %s(%d)\n", program, strerror(e),e);
		return EXIT_FAILURE;
	}

	dirwatchid = inotify_add_watch(notifd, directory, IN_MOVED_TO|IN_CLOSE_WRITE|IN_ONLYDIR|IN_DELETE_SELF);
	if( dirwatchid < 0 ) {
		e = errno;
		fprintf(stderr, "%s: Error adding watch for %s: %s(%d)\n", program, directory, strerror(e), e);
		close(notifd);
		return EXIT_FAILURE;
	}

	if( initialsearch ) {
		dir = opendir(directory);
		if( dir == NULL ) {
			e = errno;
			fprintf(stderr, "%s: Error opening directory %s: %s(%d)\n", program, directory, strerror(e), e);
			close(notifd);
			return EXIT_FAILURE;
		}
	} else
		dir = NULL;

	if( pidfilename != NULL ) {
		pidfile = fopen(pidfilename, "w");
		if( pidfile == NULL ) {
			e = errno;
			logerror_die("Could not create/open pidfile '%s': %s(%d)\n",
					pidfilename, strerror(e), e);
		}
	} else
		pidfile = NULL;
	if( !foreground ) {
		if( logfilename != NULL ) {
			logfile = fopen(logfilename, "a");
			if( logfile == NULL ) {
				e = errno;
				logerror_die("Error opening logfile '%s': %s(%d)\n",
						logfilename, strerror(e), e);
			}
			free(logfilename);logfilename = NULL;
			setlinebuf(logfile);
		} else
			openlog("queueprocessor", LOG_PID, LOG_DAEMON);
		if( daemon(false, false) < 0 ) {
			e = errno;
			logerror_die("Failed to daemonize: %s(%d)\n",
					strerror(e), e);
		}
		amdaemon = true;
	} else {
		setlinebuf(stdout);
		setlinebuf(stderr);
	}
	if( pidfile != NULL ) {
		fprintf(pidfile, "%llu", (unsigned long long)getpid());
		fclose(pidfile);
	}
	closeallfds(notifd, (logfile==NULL)?-1:fileno(logfile),
			    (dir==NULL)?-1:dirfd(dir), true);
	signal(SIGTERM, termsignal);
	signal(SIGABRT, termsignal);

	if( dir != NULL ) {
		process_directory(dir);
		closedir(dir);
	}

	timeout = -1;
	while( !termsignaled ) {
		if( waitforevent(notifd, timeout) ) {
			bool doread = true;
			if( termsignaled )
				break;
			while( (event = getevent(notifd, &doread)) != NULL ) {
				processevent(event);
				free(event);
			}
		} else {
			timeout = -1;
		}
	}
	close(notifd);
	return EXIT_SUCCESS;
}
