// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __gnu_c_tree_h__
#define __gnu_c_tree_h__

namespace Puma {

// Syntax tree node hierarchy:
class   CT_GnuAsmDef;        // derived from CT_AsmDef
class   CT_GnuAsmOperand;    // derived from CTree
class     CT_GnuAsmOperands; // derived from CT_List
class     CT_GnuAsmClobbers; // derived from CT_List
class   CT_GnuStatementExpr; // derived from CT_Expression

} // namespace Puma

#include "Puma/CTree.h"

namespace Puma {


class CT_GnuAsmDef : public CT_AsmDef {
  CTree *_cvqual;
  CTree *_operands0;
  CTree *_operands1;
  CTree *_clobbers;

public:

  CT_GnuAsmDef (CTree *a, CTree *cv, CTree *o, CTree *s, 
                CTree *op0, CTree *op1, CTree *cl, CTree *c, CTree *sc) :
    CT_AsmDef (a, o, s, c, sc) {
      _cvqual = cv; _operands0 = op0; _operands1 = op1; _clobbers = cl;
  }

  static const char *NodeId ();
  const char *NodeName () const { return NodeId (); }  

  virtual int Sons () const {
    return 5 +
           (_cvqual ? 1 : 0) +
           (_operands0 ? 1 : 0) +
           (_operands1 ? 1 : 0) +
           (_clobbers ? 1 : 0);
  }

  virtual CTree *Son (int n) const {
    int have_cv = _cvqual ? 1 : 0;
    int gnu_sons = Sons () - have_cv - 5;
    int gnu_first = 3 + have_cv;
    if (n == 0) return CT_AsmDef::Son (0);
    else if (n == 1 && _cvqual) return _cvqual;
    else if ((n == 1 && !_cvqual) || (n == 2 && _cvqual))
       return CT_AsmDef::Son (1);
    else if ((n == 2 && !_cvqual) || (n == 3 && _cvqual))
       return CT_AsmDef::Son (2);
    else if (n == gnu_first && gnu_sons >= 1) return _operands0;
    else if (n == gnu_first + 1 && gnu_sons >= 2) return _operands1;
    else if (n == gnu_first + 2 && gnu_sons >= 3) return _clobbers;
    else if (n == Sons () - 2)
       return CT_AsmDef::Son (3);
    else if (n == Sons () - 1)
       return CT_AsmDef::Son (4);
    else return (CTree*)0;
  }

  virtual void ReplaceSon (CTree *old_son, CTree *new_son) {
    if (old_son == _cvqual) _cvqual = new_son;
    else if (old_son == _operands0) _operands0 = new_son;
    else if (old_son == _operands1) _operands1 = new_son;
    else if (old_son == _clobbers) _clobbers = new_son;
    else CT_AsmDef::ReplaceSon (old_son, new_son);
  }
};

class CT_GnuAsmOperand : public CTree {
  CTree *_string;
  CTree *_open;   // CT_Token
  CTree *_expr;
  CTree *_close;  // CT_Token

public:
  CT_GnuAsmOperand (CTree *s, CTree *o, CTree *e, CTree *c) :
    _string (s), _open (o), _expr (e), _close (c) {}
  static const char *NodeId ();
  const char *NodeName () const { return NodeId (); } 
  int Sons () const { return 4; }
  CTree *Son (int n) const {
    switch (n) {
      case 0: return _string;
      case 1: return _open;
      case 2: return _expr;
      case 3: return _close;
      default: return (CTree*)0;
    }
  }
  CT_Expression *Expr () const { return (CT_Expression*)_expr; }
  CT_String *String () const { return (CT_String*)_string; }
  void ReplaceSon (CTree *old_son, CTree *new_son) {
    if (old_son == _expr) _expr = new_son;
    else if (old_son == _string) _string = new_son;
    else if (old_son == _open) _open = new_son;
    else if (old_son == _close) _close = new_son;
  }
};

class CT_GnuAsmOperands : public CT_List {
public:
  CT_GnuAsmOperands () { AddProperties (OPEN | SEPARATORS); }
  static const char *NodeId ();
  const char *NodeName () const { return NodeId (); }  
};

class CT_GnuAsmClobbers : public CT_List {
public:
  CT_GnuAsmClobbers () { AddProperties (OPEN | SEPARATORS); }
  static const char *NodeId ();
  const char *NodeName () const { return NodeId (); }  
};

/*****************************************************************************/
/*                                                                           */
/*                              Expressions                                  */
/*                                                                           */
/*****************************************************************************/

class CT_GnuStatementExpr : public CT_Expression {
  CTree *_open;
  CTree *_stmt;
  CTree *_close;
public:
  CT_GnuStatementExpr (CTree *o, CTree *s, CTree *c) :
    _open (o), _stmt (s), _close (c) {}
  static const char *NodeId ();
  const char *NodeName () const { return NodeId (); }  
  int Sons () const { return 3; }
  CTree *Son (int n) const {
    switch (n) {
      case 0: return _open;
      case 1: return _stmt;
      case 2: return _close;
      default: return (CTree*)0;
    }
  }
  CT_CmpdStmt *CmpdStmt () const { return (CT_CmpdStmt*)Son (1); }
};

/*****************************************************************************/
/*                                                                           */
/*                         Declaration specifiers                            */
/*                                                                           */
/*****************************************************************************/

// typeof(expr) or typeof(named type) feature
class CT_GnuTypeof : public CT_DeclSpec, public CSemValue {
  CTree *sons[5]; // key, open, type, close, expr

public:
  CT_GnuTypeof (CTree *k, CTree *o, CTree *t, CTree *c) {
    sons[0] = k; sons[1] = o; sons[2] = t; sons[3] = c; sons[4] = 0;
  }
  CT_GnuTypeof (CTree *k, CTree *e) {
    sons[0] = k; sons[1] = 0; sons[2] = 0; sons[3] = 0; sons[4] = e;
  }
  static const char *NodeId ();
  const char *NodeName () const { return NodeId (); }
  int Sons () const { return CTree::Sons (sons, 5); }
  CTree *Son (int n) const { return CTree::Son (sons, 5, n); }
  void ReplaceSon (CTree *old_son, CTree *new_son) { 
    CTree::ReplaceSon (sons, 5, old_son, new_son);
  }
  CTree *Expr () const { return sons[4]; }
  CT_NamedType *TypeName () const { return (CT_NamedType*)sons[2]; }
  CSemValue *SemValue () const { return (CSemValue*)this; }
  CTypeInfo *Type () const { return type; }
};

} // namespace Puma

#endif /* __gnu_c_tree_h__ */
