// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/OptsParser.h"
#include "Puma/ParserKey.h"
#include <string>
//#include <iostream>
using namespace std;

namespace Puma {


OptsParser::OptsParser (int argc, char **argv, const Option *opts) : m_opts (opts) {
  for (int idx = 1; idx < argc; idx++) {
    m_argv.push_back (argv[idx]);
  }
  m_curarg = m_tonext = m_number = 0;
}


OptsParser::OptsParser (const string &line, const Option *opts) : m_opts (opts) {
  tokenize (line, m_argv, " \t\n\r");
  m_curarg = m_tonext = m_number = 0;
}


int OptsParser::getOption () {
  m_number = FINISH;

  m_curarg += m_tonext;
  if (m_curarg < m_argv.size ()) {
    m_tonext=1;
    m_opt = m_arg = "";
    const string &arg (m_argv[m_curarg]);
    if (ParserKey::isLongKey (arg) && arg.length () > ParserKey::getLongKeyLength ()) {
      m_number = getLongOption (arg);
    } else if (ParserKey::isKey (arg) && arg.length () > 1) {
      m_number = getShortOption (arg);
    } else {
//cout << "no option: " << arg << endl;
      m_arg = strip (arg);
      m_number = NOOPTION;
    }
  }

  return m_number;
}


int OptsParser::getLongOption (const string &arg) {
  int res = UNKNOWN;
  const Option *opts = m_opts;
  unsigned int keyLen = ParserKey::getLongKeyLength ();

  // search for long option name in arg	
  while (opts->number != 0) {
    if (opts->name != NULL) {
      if ((opts->args != AT_NONE) || 
          ((arg.length () - keyLen) == strlen (opts->name))) {
        if (arg.find (opts->name) == keyLen)
          break;
      }
    }
    opts++;
  }
  
  // if we found an option go on with processing
  if (opts->number) {
    res = opts->number;

    // option possibly has arguments
    if (opts->args != AT_NONE) {
      if (processOption (arg, (unsigned int)strlen (opts->name) + 
                         ParserKey::getLongKeyLength (),opts->args) == false)
        res = NOARG;
    } else 
      m_opt = arg.substr (0, strlen (opts->name) + ParserKey::getLongKeyLength ());
//cout << "found long option: " << arg << " with argument " << getArgument () << endl;
  } else {
//cout << "unknown long option: " << arg << endl;
    m_arg = arg;
  }

  return res;
}


int OptsParser::getShortOption (const string &arg) {
  int res = UNKNOWN;
  const Option *opts = m_opts;

  // search for short option
  while (opts->number != 0 && opts->key != arg[1]) {
    opts++;
  }
  
  // ok we found a short option that fits
  if (opts->number) {
    res = opts->number;
    
    if (processOption (arg, 2,opts->args) == false){
        res = NOARG;
    }
    
//cout << "found short option: " << arg << " with argument " << getArgument () << endl;
    
  } else {
//cout << "unknown short option: " << arg << endl;
    m_arg = arg;
  }
        
  return res;
}

// process option and option argument.
// there can AT_NONE, a AT_MANDATORY or an AT_OPTIONAL option argument. 
bool OptsParser::processOption (const string &arg, unsigned int len,ArgType argtype) {
  
  // set current option
  m_opt = arg.substr (0, len);
  
  // if there should be no argument provided don't process it
  if (argtype == AT_NONE){
  	  return true;
  }	   
  
  // if there are characters behind the end of the option name, 
  // treat them as option argument
  if (arg.length () > len) {
	  m_arg = strip (arg.substr (len, arg.length () - len));
  } 

  // if ARGV contains at least one more element
  // check if it is an option argument
  else if (m_curarg + 1 < m_argv.size () ) {

	  // if the next element of ARGV is NOT an other option treat it as option argument 
	  // otherwise m_arg will continue containing an empty string
	  const string &next_argv(m_argv[m_curarg + 1]);
	  if (! ( (ParserKey::isLongKey(next_argv) && next_argv.length() > ParserKey::getLongKeyLength()) ||   
			  (ParserKey::isKey(next_argv)     && next_argv.length() > ParserKey::getKeyLength())))
	  {
		  m_arg = strip(next_argv);
		  ++m_tonext;
	  }

  }

  // return false if there should be an argument
  if( m_arg.empty() && argtype == AT_MANDATORY){
  	return false;	
  }

  return true;
}


bool OptsParser::revokeArgument () {
  bool ret=false;
  if (m_tonext > 1){
	 --m_tonext;
	 ret=true;
  }
  return ret;
}

int OptsParser::getCurrentArgNum () const {
  return m_curarg + 1;
}

int OptsParser::getNextArgNum () const {
  return m_curarg + m_tonext + 1 ;
}


int OptsParser::getResult () const {
  return m_number;
}


const string &OptsParser::getArgument () const {
  return m_arg;
}


const string &OptsParser::getOptionName () const {
  return m_opt;
}


} // namespace Puma
