// fragment.cc
//
//   Copyright 2004 Daniel Burrows

#include "fragment.h"

#include "config/colors.h"

#include <stdarg.h>

#include <algorithm>

using namespace std;

fragment::~fragment()
{
}

/** A fragment of text, possibly with attached attributes. */
class _text_fragment:public fragment
{
public:
  _text_fragment(const chstring &_s):s(_s) {}

  fragment_contents layout(size_t w)
  {
    fragment_contents rval;
    rval.push_back(s);
    return rval;
  }

  void set_attr(int attr)
  {
    s.set_attr(attr);
  }
private:
  chstring s;
};

fragment *text_fragment(const chstring &s) { return new _text_fragment(s); }
fragment *text_fragment(const string &s,
			int attr)
{
  return new _text_fragment(chstring(s, attr));
}

/** This fragment just expands to a newline. */
class _newline_fragment:public fragment
{
public:
  _newline_fragment() {}

  fragment_contents layout(size_t w)
  {
    fragment_contents rval;
    rval.set_final_nl(true);
    return rval;
  }

  void set_attr(int attr)
  {
  }
};

fragment *newline_fragment()
{
  return new _newline_fragment;
}

fragment *attr_fragment(fragment *f,
			int attr)
{
  f->set_attr(attr);
  return f;
}

/** A fragment generated by composing a sequence of other fragments. */
class _sequence_fragment:public fragment
{
public:
  _sequence_fragment(const vector<fragment*> &_contents):contents(_contents) {}

  fragment_contents layout(size_t w)
  {
    fragment_contents rval;

    rval.push_back(string(""));

    for(vector<fragment*>::const_iterator i=contents.begin();
	i!=contents.end(); ++i)
      {
	fragment_contents lines=(*i)->layout(w);

	// Make sure that implicit newlines are handled correctly.
	if(lines.size()==0)
	  {
	    if(rval.get_final_nl() && lines.get_final_nl())
	      rval.push_back(chstring(""));

	    rval.set_final_nl(rval.get_final_nl() || lines.get_final_nl());
	  }

	for(fragment_contents::const_iterator j=lines.begin();
	    j!=lines.end(); ++j)
	  {
	    if(!rval.get_final_nl())
	      {
		rval.back()+=*j;
		rval.set_final_nl(true);
	      }
	    else
	      rval.push_back(*j);
	  }

	rval.set_final_nl(lines.get_final_nl());
      }

    return rval;
  }

  void set_attr(int attr)
  {
    for(vector<fragment*>::const_iterator i=contents.begin();
	i!=contents.end(); ++i)
      (*i)->set_attr(attr);
  }

  ~_sequence_fragment()
  {
    for(vector<fragment*>::const_iterator i=contents.begin();
	i!=contents.end(); ++i)
      delete *i;
  }

private:
  vector<fragment*> contents;
};

fragment *sequence_fragment(const vector<fragment*> &contents)
{
  return new _sequence_fragment(contents);
}

fragment *sequence_fragment(fragment *f, ...)
{
  vector<fragment*> fragments;

  if(f==NULL)
    return new _sequence_fragment(fragments);

  fragments.push_back(f);

  va_list lst;

  va_start(lst, f);
  do
    {
      f=va_arg(lst, fragment*);
      if(f!=NULL)
	fragments.push_back(f);
    } while(f!=NULL);

  va_end(lst);

  return new _sequence_fragment(fragments);
}

// Fall back on sequence_fragment [for now?]
fragment *join_fragments(const vector<fragment *> &fragments,
			 const string &between, int attr)
{
  vector<fragment*> rval;

  for(vector<fragment *>::const_iterator i=fragments.begin();
      i!=fragments.end();
      ++i)
    {
      if(i!=fragments.begin())
	rval.push_back(text_fragment(between, attr));

      rval.push_back(*i);
    }

  return sequence_fragment(rval);
}

class _flowbox:public fragment
{
public:
  _flowbox(fragment *_contents):contents(_contents) {}

  fragment_contents layout(size_t w)
  {
    if(w==0)
      return fragment_contents();

    fragment_contents rval,lines=contents->layout(w);

    for(fragment_contents::const_iterator i=lines.begin();
	i!=lines.end(); ++i)
      {
	chstring s=*i;

	// Make sure we at least advance the cursor for every line read.
	//
	// Is there a less gross way to express this?
	bool output_something=false;

	size_t first=0;

	// Flow the current line:
	while(first<s.size())
	  {
	    // Strip leading whitespace.
	    while(first<s.size() && isspace(s[first]&A_CHARTEXT))
	      ++first;

	    if(first==s.size())
	      break;

	    // If there are few enough characters to fit on a single
	    // line, do that.
	    if(s.size()-first<=w)
	      {
		rval.push_back(chstring(s, first, s.size()-first));
		first=s.size();
		output_something=true;
	      }
	    else
	      {
		size_t amt=w;

		while(amt>0 && !isspace(s[first+amt]&A_CHARTEXT))
		  --amt;

		// Oops, there's a word that's longer than the current line.
		if(amt==0)
		  {
		    rval.push_back(chstring(s, first, w));
		    output_something=true;
		    first+=w;
		  }
		else
		  {
		    // Eh, strip trailing whitespace.
		    while(amt>0 &&
			  isspace(s[first+amt]&A_CHARTEXT) &&
			  isspace(s[first+amt-1]&A_CHARTEXT))
		      --amt;

		    rval.push_back(chstring(s, first, amt));
		    first+=amt;
		    output_something=true;
		  }
	      }
	  }

	if(!output_something)
	  rval.push_back(chstring(""));

	// Ok, go to the next line now.
      }

    // flowboxes always have a final newline.
    rval.set_final_nl(true);

    return rval;
  }

  void set_attr(int attr) { contents->set_attr(attr); }

  ~_flowbox() { delete contents; }

private:
  fragment *contents;
};

fragment *flowbox(fragment *contents) {return new _flowbox(contents);}

class _fillbox:public fragment
{
public:
  _fillbox(fragment *_contents):contents(_contents) {}

  fragment_contents layout(size_t w)
  {
    if(w==0)
      return fragment_contents();

    fragment_contents rval,lines=contents->layout(w);

    for(fragment_contents::const_iterator i=lines.begin();
	i!=lines.end(); ++i)
      {
	chstring s=*i;

	size_t first=0;

	// Build a list of words on the current line.
	vector<chstring> words;

	bool output_something=false;

	while(first<s.size())
	  {
	    // Strip leading whitespace.
	    while(first<s.size() && isspace(s[first]&A_CHARTEXT))
	      ++first;

	    size_t amt=0;
	    while(first+amt<s.size() && !isspace(s[first+amt]&A_CHARTEXT))
	      ++amt;

	    if(amt>0)
	      words.push_back(chstring(s, first, amt));

	    first+=amt;
	  }

	// Now place them onto output lines.

	size_t word_start=0;

	while(word_start<words.size())
	  {
	    size_t curwidth=0;
	    size_t nwords=0;

	    // As long as adding the *next* word doesn't put us
	    // past the right edge, add it.
	    while(word_start+nwords < words.size() &&
		  curwidth+words[word_start+nwords].size()+nwords <= w)
	      {
		curwidth=curwidth+words[word_start+nwords].size();
		++nwords;
	      }

	    if(nwords==0)
	      {
		// Split a single word: just chop the beginning off.
		rval.push_back(chstring(words[word_start], 0, w));
		words[word_start]=chstring(words[word_start], w);
		output_something=true;
	      }
	    else
	      {
		size_t diff;

		if(word_start+nwords<words.size())
		  diff=w-(curwidth+nwords-1);
		else
		  // Cheat to disable fililng on the last line of the
		  // paragraph.
		  diff=0;

		// Now spit the words into an output string, filled
		// left and right.
		chstring final("");

		// This is similar to the famous algorithm for drawing
		// a line.  The idea is to add diff/(words-1) spaces
		// (in addition to one space per word); since
		// fractional spaces aren't allowed, I approximate by
		// adding a number of spaces equal to the integral
		// part, then keeping the remainder for the next word.
		size_t extra_spaces=0;

		for(size_t word=0; word<nwords; ++word)
		  {
		    if(word>0)
		      // Insert spaces between words:
		      {
			extra_spaces+=diff;

			size_t nspaces=1+extra_spaces/(nwords-1);
			extra_spaces%=nwords-1;

			final+=chstring(nspaces, ' ');
		      }

		    final+=words[word+word_start];
		  }

		output_something=true;
		rval.push_back(final);

		word_start+=nwords;
	      }
	  }

	if(!output_something)
	  rval.push_back(chstring(""));
      }

    // fillboxes always have a final newline.
    rval.set_final_nl(true);

    return rval;
  }

  void set_attr(int attr) { contents->set_attr(attr); }

  ~_fillbox() { delete contents; }

private:
  fragment *contents;
};

fragment *fillbox(fragment *contents) {return new _fillbox(contents);}

class _clipbox:public fragment
{
public:
  _clipbox(fragment *_contents):contents(_contents) {}

  fragment_contents layout(size_t w)
  {
    fragment_contents rval, lines=contents->layout(w);

    for(fragment_contents::const_iterator i=lines.begin(); i!=lines.end(); ++i)
      {
	if(i->size()<=w)
	  rval.push_back(*i);
	else
	  rval.push_back(chstring(*i, 0, w));
      }

    // Clipboxes are always followed by a final newline.
    rval.set_final_nl(true);

    return rval;
  }

  void set_attr(int attr) { contents->set_attr(attr); }

  ~_clipbox() { delete contents; }

private:
  fragment *contents;
};

fragment *clipbox(fragment *contents) {return new _clipbox(contents);}

class _indentbox:public fragment
{
public:
  _indentbox(size_t _firstindent, size_t _restindent, fragment *_contents)
    :contents(_contents), indentattr(0),
     firstindent(_firstindent), restindent(_restindent) {}

  fragment_contents layout(size_t w)
  {
    if(w<=firstindent || w<=restindent)
      return fragment_contents();

    fragment_line firstprepend(firstindent, ' '|indentattr);
    fragment_line restprepend(restindent, ' '|indentattr);

    fragment_contents rval, lines=contents->layout(w-restindent);

    for(fragment_contents::const_iterator i=lines.begin(); i!=lines.end(); ++i)
      {
	chstring l=((i==lines.begin())?firstprepend:restprepend)+*i;

	rval.push_back(l);
      }

    // Indentboxes are always followed by a final newline.
    rval.set_final_nl(true);

    return rval;
  }

  void set_attr(int attr) { contents->set_attr(attr); indentattr=attr; }

  ~_indentbox() { delete contents; }
private:
  fragment *contents;

  int indentattr;

  size_t firstindent, restindent;
};

fragment *indentbox(size_t firstindent,
		    size_t restindent,
		    fragment *contents)
{
  return new _indentbox(firstindent, restindent, contents);
}

struct argument
{
  argument():format(0) {}

  char format;
  union
  {
    fragment *F;
    const char *s;
  };
};

fragment *fragf(const char *format, ...)
{
  int argcount=0;
  int posargcount=0;

  const char *start=format;
  // find all the arguments.
  char *nextpercent=strchr(start, '%');

  // loop 1: count the arguments.
  while(nextpercent!=NULL)
    {
      switch(*(nextpercent+1))
	{
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	  {
	    char *endptr;

	    int pos=strtol(nextpercent+1, &endptr, 10);

	    if(*endptr!='$' || (*(endptr+1)!='F' && *(endptr+1)!='s'))
	      return text_fragment("Internal error: bad format string",
				   get_color("Error"));

	    posargcount=max(posargcount, pos);

	    start=endptr+2;
	  }
	  break;

	case 'B':
	case 'b':
	case 'R':
	case 'r':
	case 'D':
	case 'd':
	case 'n':
	case '%':
	  start=nextpercent+2;
	  break;
	case 's':
	case 'F':
	  ++argcount;
	  start=nextpercent+2;
	  break;
	default:
	  return text_fragment("Internal error: bad format string",
			       get_color("Error"));
	}

      nextpercent=strchr(start, '%');
    }

  int nargs=max(argcount, posargcount);

  argument arguments[nargs];

  argcount=0;

  // loop 2: read the list of arguments and parse their type.
  start=format;
  nextpercent=strchr(start, '%');
  while(nextpercent!=NULL)
    {
      switch(*(nextpercent+1))
	{
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	  {
	    char *endptr;

	    int pos=strtol(nextpercent+1, &endptr, 10)-1;

	    // if we saw it before it had better be the same type.
	    if(arguments[pos].format!=0 && arguments[pos].format!=*(endptr+1))
	      return text_fragment("Bad argument string to fragf: inconsistent positional parameter types!",
				   get_color("Error"));

	    arguments[pos].format=*(endptr+1);

	    start=endptr+2;
	  }
	  break;

	case 'B':
	case 'b':
	case 'R':
	case 'r':
	case 'D':
	case 'd':
	case 'n':
	case '%':
	  start=nextpercent+2;
	  break;
	case 's':
	case 'F':
	  if(arguments[argcount].format!=0 && arguments[argcount].format!=*(nextpercent+1))
	    return text_fragment("Bad argument string to fragf: inconsistent parameter types!",
				 get_color("Error"));

	  arguments[argcount].format=*(nextpercent+1);

	  ++argcount;
	  start=nextpercent+2;
	  break;
	default:
	  return text_fragment("Internal error: bad format string",
			       get_color("Error"));
	}

      nextpercent=strchr(start, '%');
    }


  int attr=A_NORMAL;

  va_list arglst;
  va_start(arglst, format);

  for(int i=0; i<nargs; ++i)
    {
      switch(arguments[i].format)
	{
	case 0:
	  break; // I suppose unused arguments are ok
	case 'F':
	  arguments[i].F=va_arg(arglst, fragment *);
	  break;
	case 's':
	  arguments[i].s=va_arg(arglst, const char *);
	  break;
	default:
	  return text_fragment("Internal error: bad format string",
			       get_color("Error"));
	}
    }

  va_end(arglst);

  // Now parse this and generate a result.
  vector<fragment *> rval;

  start=format;
  nextpercent=strchr(start, '%');

  // Current argument for non-positional arguments.
  argcount=0;

  // Optimization: don't create lots of unnecessary text fragments
  // when one will do.
  chstring curstr("");

  // Loop 3: execute the program

  while(nextpercent!=NULL)
    {
      // First, make a fragment for everything until the percent:
      curstr+=chstring(string(start, nextpercent-start), attr);

      // This is almost always what we want; in the cases when it's not,
      // I override it explicitly.
      start=nextpercent+2;

      // Now, find what's at the percent.
      switch(*(nextpercent+1))
	{
	case '%':
	  curstr+=chstring(string("%"), attr);
	  break;
	case 'B':
	  attr|=A_BOLD;
	  break;
	case 'b':
	  attr&=~A_BOLD;
	  break;
	case 'R':
	  attr|=A_REVERSE;
	  break;
	case 'r':
	  attr&=~A_REVERSE;
	  break;
	case 'D':
	  attr|=A_DIM;
	  break;
	case 'd':
	  attr&=~A_DIM;
	  break;
	case 'n':
	  if(!curstr.empty())
	    {
	      rval.push_back(text_fragment(curstr));
	      curstr="";
	    }

	  rval.push_back(newline_fragment());
	  break;
	case 'F':
	  // should have been verified above.
	  assert(arguments[argcount].format=='F');

	  if(!curstr.empty())
	    {
	      rval.push_back(text_fragment(curstr));
	      curstr="";
	    }
	  rval.push_back(arguments[argcount].F);
	  ++argcount;
	  break;
	case 's':
	  // should have been verified above.
	  assert(arguments[argcount].format=='s');

	  curstr+=chstring(arguments[argcount].s, attr);
	  ++argcount;
	  break;
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	  {
	    char *endptr;

	    int pos=strtol(nextpercent+1, &endptr, 10)-1;

	    assert(arguments[pos].format==*(endptr+1));

	    switch(*(endptr+1))
	      {
	      case 'F':
		if(!curstr.empty())
		  {
		    rval.push_back(text_fragment(curstr));
		    curstr="";
		  }

		rval.push_back(arguments[pos].F);
		break;
	      case 's':
		curstr+=chstring(arguments[pos].s, attr);
		break;
	      }

	    start=endptr+2;
	    break;
	  }
	}

      nextpercent=strchr(start, '%');
    }

  // Get any trailing bit o' string:
  if(*start!=0)
    curstr+=chstring(start, attr);

  if(!curstr.empty())
    rval.push_back(text_fragment(curstr));

  return sequence_fragment(rval);
}
