#
# Copyright (C) 2005 Chris Halls <halls@debian.org>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of version 2.1 of the GNU Lesser General Public
# License as published by the Free Software Foundation.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Unit test for packages.py"""

from apt_proxy import packages 
#import AptPackages, PackageFileList, get_mirror_versions, AptDpkgInfo, cleanup
from apt_proxy.apt_proxy import Factory
from apt_proxy.test.test_apt_proxy import apTestHelper, FactoryTestHelper
from apt_proxy.apt_proxy_conf import apConfig
from StringIO import StringIO
from twisted.trial import unittest
import shutil, os, re, glob

class DummyCacheEntry:
    """
    Class that provides basic CacheEntry information
    """
    def __init__(self, cache_dir, backend, file):
        self.filename = os.path.basename(file)
        self.path = file
        self.cache_path = backend + os.sep + file
        self.file_path = cache_dir + os.sep + self.cache_path
    def get_request_mtime(self):
        return None

class PackageFileListTest(apTestHelper):
    def setUp(self):
        apTestHelper.setUp(self)
        self.f = packages.PackageFileList('test', self.cache_dir)
    def tearDown(self):
        del(self.f) # Needed otherwise we'll get a database exception when cache dir is removed
        apTestHelper.tearDown(self)
    def testEmpty(self):
        self.assertEqual(self.f.get_files(),[])
    def testAddPackages(self):
        shutil.copytree('../test_data/packages', self.cache_dir+'/packages')
        self.f.update_file(DummyCacheEntry(self.cache_dir, 'packages','Packages'))
        self.assertEqual(self.f.get_files(),['packages/Packages'])
        self.f.update_file(DummyCacheEntry(self.cache_dir, 'packages','Packages.gz')) # This file should not be added
        self.assertNotIn('packages/Packages.gz', self.f.get_files())

        # Remove packages file and check that it is removed from database
        os.unlink(self.cache_dir+'/packages/Packages')
        self.assertEqual(self.f.get_files(),[])

class PackagesCacheTest(apTestHelper):
    def setUp(self):
        apTestHelper.setUp(self)
        self.p = packages.AptPackages('test', self.cache_dir)
    def tearDown(self):
        del(self.p)
        apTestHelper.tearDown(self)
    def testLoadEmpty(self):
        self.failIfEqual(self.p.load(),True)
    def testReLoadEmpty(self):
        self.failIfEqual(self.p.load(),True)
        del(self.p)
        self.p = packages.AptPackages('test', self.cache_dir)

class PackagesTestHelper(apTestHelper):
    """
    Create a test AptPackages using test data
    """
    def setUp(self):
        apTestHelper.setUp(self)
        self.p = packages.AptPackages('test1', self.cache_dir)
        shutil.copytree('../test_data/packages', self.cache_dir+'/packages')
        self.p.file_updated(DummyCacheEntry(self.cache_dir, 'packages','Packages'))
        #print "Cache dir:", self.cache_dir, '\n'
    def tearDown(self):
        del(self.p)
        apTestHelper.tearDown(self)

class MirrorVersionsTest(PackagesTestHelper):
    def testGetMirrorVersions(self):
        self.assertEquals(self.p.load(),True)
        # Test that an unknown package returns empty version set
        self.assertEqual(self.p.get_mirror_versions('nonexistent'), [])
        # Test for apt package from test_data directory
        aptver = get_test_deb_info()['Version']
        self.assertNotEquals(re.match('[0-9.a-z]+', aptver), None)

class ImportTest(FactoryTestHelper):
    import_config="""
[DEFAULT]
debug=all:9 apt:0
port=9999
address=
cleanup_freq=off
max_versions=off

[other]
backends = http://a.b.c/d

[packages]
backends = http://a.b.c/d

"""

    def setUp(self):
        FactoryTestHelper.setUp(self, self.import_config)
        self.packages_dir = self.cache_dir+'/packages'
        self.source_dir = '../test_data/packages'
        self.p = packages.AptPackages('packages', self.cache_dir)
        os.mkdir(self.packages_dir)
        shutil.copyfile(self.source_dir + '/Packages', self.packages_dir + '/Packages')
        # Register test package files in db
        entry = DummyCacheEntry(self.cache_dir,'packages','Packages')
        self.factory.getBackend('packages').get_packages_db().file_updated(entry)
    def tearDown(self):
        del self.p
    def testImport(self):
        globpattern = self.packages_dir + '/apt_*_*.deb' # The file we expect to import
        self.assertEquals(len(glob.glob(globpattern)), 0)
        self.assertEquals(packages.import_directory(self.factory, self.source_dir, False), 1)
        self.assertEquals(len(glob.glob(globpattern)), 1)

    def testImportEmpty(self):
        self.assertEquals(packages.import_directory(self.factory, '../test_data', False), 0)
    def testImportRecursive(self):
        self.assertEquals(packages.import_directory(self.factory, '../test_data', True), 3)

def get_test_deb_name():
    "Return filename of test deb file"
    debs = glob.glob('../test_data/packages/apt_*_*.deb')
    return debs[-1]

def get_test_deb_info():
    "Return an AptDpkgInfo for our test deb"
    return packages.AptDpkgInfo(get_test_deb_name())

class AptDpkgInfoTest(unittest.TestCase):
    def testGetInfo(self):
        info = get_test_deb_info()
        self.assertEquals(info['Package'], 'apt')
        self.assertNotEquals(re.match('[0-9.a-z]+', info['Version']), None)

# Backend definition for factory.get_mirror_versions
config_mirrorvers="""
[packages]
backends=http://dummy
"""

class FactoryVersionFuncsTest(apTestHelper):
    def setUp(self):
        apTestHelper.setUp(self)
        shutil.copytree('../test_data/packages', self.cache_dir+'/packages')
        self.c = apConfig(StringIO(self.config+config_mirrorvers))
        self.factory = Factory(self.c)
        self.factory.createBackends()
        # Register test package files in db
        entry = DummyCacheEntry(self.cache_dir,'packages','Packages')
        self.factory.getBackend('packages').get_packages_db().file_updated(entry)
        # Get version of apt used for testing
        self.aptinfo = get_test_deb_info()
        #print self.cache_dir
    def tearDown(self):
        packages.cleanup(self.factory)
        del(self.factory)
        apTestHelper.tearDown(self)

    def testGetAllMirrorVersions(self):
        aptversions = packages.get_mirror_versions(self.factory, 'apt')
        self.assertEquals(self.aptinfo['Version'], aptversions[0][0])
        testdeb_name = get_test_deb_name().replace('../test_data/','') # strip test data directory
        self.assertEquals(testdeb_name, aptversions[0][1])
        self.assertEquals([], packages.get_mirror_versions(self.factory, 'unknown'))
    def testGetEmptyMirrorVersions(self):
        self.assertEquals([], packages.get_mirror_versions(self.factory, 'unknown'))
    #def testGetMirrorPath(self):
        #self.assertEquals([], packages.get_mirror_path(self.factory, 'unknown'))
